/// @addtogroup unifexp
/// @{

/////////////////////////////////////////////////////////////////////////////
///
/// @file worker.h
///
/// This file contains the definition of the worker class
/// whose objects process single chunks of data.
///
/// @author Pawel Pilarczyk
///
/////////////////////////////////////////////////////////////////////////////

// Copyright (C) 2007 by Pawel Pilarczyk.
//
// This file is part of my research program package.  This is free software;
// you can redistribute it and/or modify it under the terms of the GNU
// General Public License as published by the Free Software Foundation;
// either version 2 of the License, or (at your option) any later version.
//
// This software is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License along
// with this software; see the file "license.txt".  If not, write to the
// Free Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
// MA 02111-1307, USA.

// Started on March 8, 2007. Last revision: August 22, 2007.

#ifndef _worker_h_
#define _worker_h_

#include "chomp/system/config.h"
#include "chomp/system/textfile.h"
#include "chomp/system/timeused.h"
#include "chomp/multiwork/mw.h"

#include "graphs.h"
#include "maptypes.h"
#include "parttypes.h"


namespace unifexp {

// --------------------------------------------------
// ---------------- the worker class -----------------
// --------------------------------------------------

/// The worker class that processes single chunks of data.
class Worker: public chomp::multiwork::mwWorker
{
public:
	/// The default constructor.
	Worker (int _controlNumber, bool _local = false);

private:
	/// The control number to recognize a compatible coordinator.
	int controlNumber;

	/// Is the work being done locally? That is, in the same process
	/// as the coordinator? If so, then the global variables are not
	/// reset, such as pointbase, because this might affect the
	/// coordinator's data.
	bool local;

	/// A function for the initialization of a worker.
	int Initialize (chomp::multiwork::mwData &data);

	/// A function for processing a piece of data by a worker.
	int Process (chomp::multiwork::mwData &data);

private:
	/// An object which contains all the map objects to choose from.
	mapTypes<double> maps;

	/// An object which contains all the partition objects
	/// to choose from.
	partTypes<double> partitions;

}; /* class Worker */

// --------------------------------------------------

inline Worker::Worker (int _controlNumber, bool _local):
	controlNumber (_controlNumber), local (_local)
{
	return;
} /* Worker::Worker */

// --------------------------------------------------

inline int Worker::Initialize (chomp::multiwork::mwData &data)
{
	// ignore the received data and say it was Ok
	return chomp::multiwork::mwOk;
} /* Worker::Initialize */

inline int Worker::Process (chomp::multiwork::mwData &data)
{
	using namespace chomp::homology;
	using namespace chomp::multiwork;

	// remember the time used so far
	double timeStamp = static_cast<double> (program_time);

	// decode the number of the data pack to process
	int currentLevel = 0, currentItem = 0;
	data >> currentLevel;
	data >> currentItem;

	// say which data pack is being processed
	sout << "- " << currentLevel << ":" << currentItem << ".\n";

	// identify the map object to use
	std::string mapName;
	data >> mapName;
	mapType<double> *theMap = maps. get (mapName);
	if (!theMap)
	{
		sout << "! Cannot find the map '" << mapName << "'.\n";
		data. Reset ();
		return mwReject;
	}

	// identify the partition object to use
	std::string partName;
	data >> partName;
	partType<double> *thePart = partitions. get (partName);
	if (!thePart)
	{
		sout << "! Cannot find the partition '" <<
			partName << "'.\n";
		data. Reset ();
		return mwReject;
	}

	// the size of the partition
	int partCount = 0;
	data >> partCount;

	// the parameter range
	double paramMin = 0, paramMax = 0;
	data >> paramMin;
	data >> paramMax;
	theMap -> setParam (paramMin, paramMax);

	// other parameters of the computations
	double deltaBad = 0;
	double delta = 0;
	double resolution = 0.01;
	double lambda = 0;
	data >> delta;
	data >> resolution;
	data >> lambda;

	// what to compute and how?
	int computeDelta = 0;
	int computeDelta0 = 0;
	int computeLambda = 0;
	int computeC = 0;
	int computeLambda0 = 0;
	int rigorous = 0;
	int sparseGraph = -1;
	data >> computeDelta;
	data >> computeDelta0;
	data >> computeLambda;
	data >> computeC;
	data >> computeLambda0;
	data >> rigorous;
	data >> sparseGraph;

	// the precision for writing real numbers to the screen and log files
	int loprec = 6;
	int hiprec = 6;
	data >> loprec;
	data >> hiprec;

	// the ending control number
	int ctrl = 0;
	data >> ctrl;
	if (ctrl != controlNumber)
	{
		sout << "! Data incomplete. Rejecting it.\n";
		data. Reset ();
		return mwReject;
	}

	// remember the default precision and set a good one for 'sbug'
	int precision = std::cout. precision ();
	sbug << std::setprecision (hiprec);
	sout << std::setprecision (hiprec);

	// run one set of computations
	double logC = 0, lambda0 = 0;
	int result = 0;
	try
	{
		// compute delta unless it has been fixed already
		if (computeDelta || computeDelta0)
		{
			bool considerPaths = !!computeDelta0;
			findDeltaBisection (lambda, resolution,
				*theMap, *thePart, partCount,
				deltaBad, delta, considerPaths,
				rigorous, sparseGraph);
		}

		// compute the constants lambda, C, and lambda0
		if (computeLambda || computeC || computeLambda0)
		{
			findLambdaC (delta, *theMap, *thePart, partCount,
				computeLambda ? &lambda : 0,
				computeC ? &logC : 0,
				computeLambda0 ? &lambda0 : 0,
				rigorous, sparseGraph);
		}
	}
	catch (const char *msg)
	{
		sout << "REJECT: " << msg << '\n';
		result = -1;
	}
	catch (const std::exception &e)
	{
		sout << "REJECT: " << e. what () << '\n';
		result = -1;
	}
	catch (...)
	{
		sout << "REJECT: An unknown error occurred.\n";
		result = -1;
	}

	// restore the defaulut precision of the two major output streams
	sbug << std::setprecision (precision);
	sout << std::setprecision (precision);

	// if the result is wrong then reject this piece of data
	if (result < 0)
	{
		data. Reset ();
		return mwReject;
	}

	// say what has been computed
	sout << "= " << currentLevel << ":" << currentItem << ".\n";

	// send back the data containing the result of the processing
	data. Reset ();
	data << currentLevel;
	data << currentItem;
	data << paramMin;
	data << paramMax;
	data << partCount;
	data << deltaBad;
	data << delta;
	data << lambda;
	data << logC;
	data << lambda0;
	data << (static_cast<double> (program_time) - timeStamp);
	data << controlNumber;

	// return the result
	return mwOk;
} /* Worker::Process */


} // namespace unifexp

#endif // _worker_h_

/// @}

