/// @addtogroup unifexp
/// @{

/////////////////////////////////////////////////////////////////////////////
///
/// @file unifexp.cpp
///
/// This file contains the main program which processes the command-line
/// arguments and runs the computations.
///
/// @author Pawel Pilarczyk
///
///
/// @namespace unifexp
///
/// This namespace contains all the classes and functions defined in the
/// Uniform Expansion project files.
///
///
/// @namespace chomp
///
/// This is the top-level namespace of the CHomP library interface;
/// most classes and functions are contained in its sub-namespaces.
///
///
/// @namespace chomp::homology
///
/// This is the main namespace that contains most of the CHomP library
/// classes and functions, some of which are used in the Uniform Expansion
/// project.
///
///
/// @mainpage The Uniform Expansion Software
///
/// \section intro Introduction
///
/// This website contains the documentation of software designed and
/// programmed by Pawel Pilarczyk which accompanies the paper
/// <em>Quantitative hyperbolicity estimates in one-dimensional
/// dynamics</em> by S. Day, H. Kokubu, S. Luzzatto, K. Mischaikow, H. Oka,
/// and P. Pilarczyk.
///
/// \section License
///
/// In order to make this software freely available for wide audience,
/// it is provided here under the terms of the GNU General Public License,
/// version 2 or any later version. Please, see the
/// <a href="http://www.gnu.org/copyleft/gpl.html">GNU website</a>
/// for details.
///
/// \section modules Modules
///
/// The documentation covers two major modules. The first one, called
/// <em>unifexp</em>, contains the part of software written specifically
/// for the Uniform Expansion project. This classes and functions
/// in this module are all gathered in the <em>unifexp</em> namespace.
///
/// The second module, called
/// <em>struct</em>, contains an extract from the Computational Homology
/// Project software, or CHomP for short, which was used in this project.
/// The data structures defined in that portion of the code are gathered
/// in the CHomP namespace called <em>chomp::homology</em>.
///
/// Note that the development of some portions of the CHomP software,
/// in particular the graph algorithms included in the header file
/// <em>digraph.h</em>, was motivated by the Uniform Expansion project.
/// Moreover, it is necessary to know their features in order to be able
/// to understand the Uniform Expansion software.
/// Therefore, it is fully justified to include them in this documentation.
///
/// \section browsing Browsing
///
/// Although a lot of effort was put into making this software as transparent
/// and clear as possible, and this documentation complete, I am sure that
/// there is a lot of important information that might have been included
/// here but was not. Therefore, this documentation was generated in such
/// a way that it contains the source code which can be browsed and checked
/// in case of any doubt.
///
/// So, what are you waiting for? I wish you
/// a happy browsing!
///
/// Pawel Pilarczyk
///
/////////////////////////////////////////////////////////////////////////////

// Copyright (C) 2007 by Pawel Pilarczyk.
//
// This file is part of my research program package.  This is free software;
// you can redistribute it and/or modify it under the terms of the GNU
// General Public License as published by the Free Software Foundation;
// either version 2 of the License, or (at your option) any later version.
//
// This software is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License along
// with this software; see the file "license.txt".  If not, write to the
// Free Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
// MA 02111-1307, USA.

// Started on March 8, 2007. Last revision: August 22, 2007.

// NOTE: This program uses the BOOST, CAPD and CHomP libraries.


// the first local header file
#include "rounding.h" // (this header must be included first - BOOST)

// other local header files
#include "coord.h"
#include "worker.h"
#include "maptypes.h"
#include "parttypes.h"

// standard library headers
#include <string>
#include <vector>
#include <exception>

// selected files from the CHomP library
#include "chomp/system/config.h"
#include "chomp/system/textfile.h"
#include "chomp/system/timeused.h"
#include "chomp/multiwork/mw.h"
#include "chomp/system/arg.h"


// --------------------------------------------------
// -------------------- OVERTURE --------------------
// --------------------------------------------------

/// The title of the main program.
const char *title = "\
UniformExpansion, ver. 0.01, 11/14/2007. (C) 2007 by Pawel Pilarczyk.\n\
This is free software. No warranty. Consult 'license.txt' for details.";

/// The help information about the main program.
const char *helpinfo = "\
This is a computer program for the rigorous analysis of the uniform\n\
expansion condition as described in the paper by S. Day, H. Kokubu,\n\
S. Luzzatto, K. Mischaikow, H. Oka, and P. Pilarczyk. See the paper\n\
for details, and the attached documentation for directions.\n\
Command line arguments:\n\
-a N - set the value of the map's parameter (an obligatory argument!),\n\
-m name - use the map with the given name (default: quadratic),\n\
-p name - use partition with the given name (default: uniform),\n\
-k N - set the number of intervals, including crit. nbhd (default: 1001),\n\
-d N - fix delta (the radius of the critical neighborhood; default: 0.01),\n\
-f name - save results of computations to this file,\n\
-s N - use algorithms for sparse graphs: 0=no, 1=yes, -1=auto (default),\n\
--rigorous - run rigorous computations (with controlled rounding),\n\
--flush - flush the data written to the results file (slow),\n\
--help - show this brief help information and exit.\n\
Arguments that define what to compute (default: lambda, C, and lambda0):\n\
--lambda - compute lambda for (1) in the paper,\n\
--C - compute C for (1) in the paper (implies the computation of lambda),\n\
--lambda0 - compute lambda for (2) in the paper (implies --lambda),\n\
--delta - compute min delta s.t. lambda > 0 (does not imply --lambda),\n\
--delta0 - compute min delta s.t. lambda0 > 0 (does not imply --lambda),\n\
-l N - fix a lambda or lambda0 for --delta or --delta0 (default: 0),\n\
-r N - relative resolution for computing log delta (default: 0.01).\n\
If one of the arguments '-a', '-d', '-k', '-l' is repeated, then the\n\
computations are run for all the parameters from the given interval.\n\
-i - use full intervals of parameters (applies only for '-a'),\n\
-L N - level of initial subdivisions (default: 4); 2^n are made at start,\n\
-L N - level of final subdivisions (def: 10); 2^n subdivisions are made.\n\
Arguments for running distributed computing:\n\
-w computer:port - run a worker and connect to the given coordinator,\n\
-c [port] - run the coordinator and listen at the given port (def: 4721).";

/// Information on where to find more information about the program.
const char *helpmore = "\
For more information ask the author at http://www.pawelpilarczyk.com/.";

/// The control number for network communication. Change this number
/// whenever an incompatible version of this program is created.
const int controlNumber = 42342;


// --------------------------------------------------
// ---------------------- MAIN ----------------------
// --------------------------------------------------

/// The main function of the program.
/// Returns: 0 = Ok, -1 = Error, 1 = Help displayed, 2 = Wrong arguments.
int main (int argc, char *argv [])
{
	using namespace chomp::homology;
	using namespace chomp::multiwork;
	using namespace unifexp;

	// prepare user-configurable data
	const double infinity = 1.7e308;
	double paramMin = -infinity;
	double paramMax = -infinity;
	double deltaMin = 0;
	double deltaMax = -infinity;
	double resolution = 0.01;
	double lambdaMin = 0;
	double lambdaMax = -infinity;
	int partCountMin = 1001;
	int partCountMax = -1;
	char *mapName = 0;
	char *defMapName = const_cast<char *> ("quadratic");
	char *defMapNameRigorous = const_cast<char *> ("quadratic-intv");
	char *partName = const_cast<char *> ("uniform");
	int port = 0;
	char *work = 0;
	bool test = false;
	bool flushfile = false;
	char *filename = 0;
	bool intervals = false;
	int startLevel = 4;
	int finalLevel = 6;
	int computeDelta = 0;
	int computeDelta0 = 0;
	int computeLambda = 0;
	int computeC = 0;
	int computeLambda0 = 0;
	int rigorous = 0;
	int sparseGraph = -1;

	// interprete the command-line arguments
	arguments a;
	arg (a, "a", paramMin);
	arg (a, "a", paramMax);
	arg (a, "d", deltaMin);
	arg (a, "d", deltaMax);
	arg (a, "r", resolution);
	arg (a, "k", partCountMin);
	arg (a, "k", partCountMax);
	arg (a, "m", mapName);
	arg (a, "p", partName);
	arg (a, "f", filename);
	arg (a, "c", port, 4721);
	arg (a, "w", work);
	arg (a, "L", startLevel);
	arg (a, "L", finalLevel);
	arg (a, "l", lambdaMin);
	arg (a, "l", lambdaMax);
	arg (a, "s", sparseGraph);
	argswitch (a, "i", intervals, true);
	argswitch (a, "t", test, true);
	argswitch (a, "-test", test, true);
	argswitch (a, "-flush", flushfile, true);
	argswitch (a, "-delta0", computeDelta0, 1);
	argswitch (a, "-delta", computeDelta, 1);
	argswitch (a, "-lambda0", computeLambda0, 1);
	argswitch (a, "-lambda", computeLambda, 1);
	argswitch (a, "-C", computeC, 1);
	argswitch (a, "-rigorous", rigorous, 1);
	arghelp (a);

	argstreamprepare (a);
	int argresult = a. analyze (argc, argv);
	argstreamset ();

	// show the program's main title
	if (argresult >= 0)
		sout << title << '\n';

	// if something was incorrect, show an additional message and exit
	if (argresult < 0)
	{
		sout << "Call with '--help' for help.\n";
		return 2;
	}

	// if help requested, show help information
	if (((paramMin == -infinity) && !work && !test) || (argresult > 0))
	{
		sout << helpinfo << '\n';
		typedef std::vector<std::string> strings;

		// show available partition types
		strings partNames;
		partTypes<double> partitions;
		partitions. getNames (partNames);
		sout << "Available partitions:";
		int pos = 14;
		for (strings::iterator it = partNames. begin ();
			it != partNames. end (); ++ it)
		{
			if (pos + (*it). length () > 70)
			{
				sout << "\n\t";
				pos = 0;
			}
			else if (pos > 0)
				sout << " ";
			sout << (*it);
			pos += (*it). length () + 1;
		}
		sout << ".\n";

		// show available map types
		strings mapNames;
		mapTypes<double> maps;
		maps. getNames (mapNames);
		sout << "Available maps:";
		pos = 8;
		for (strings::iterator it = mapNames. begin ();
			it != mapNames. end (); ++ it)
		{
			if (pos + (*it). length () > 70)
			{
				sout << "\n\t";
				pos = 0;
			}
			else if (pos > 0)
				sout << " ";
			sout << (*it);
			pos += (*it). length () + 1;
		}
		sout << ".\n";

		sout << helpmore << '\n';
		return 1;
	}

	// make sure delta is positive unless it is to be computed
	if (!computeDelta && !computeDelta0 && (deltaMin <= 0))
		deltaMin = deltaMax = 0.01;

	// determine which variable is to be iterated
	if (paramMax < paramMin)
		paramMax = paramMin;
	if (deltaMax < deltaMin)
		deltaMax = deltaMin;
	if (lambdaMax < lambdaMin)
		lambdaMax = lambdaMin;
	if (partCountMax < partCountMin)
		partCountMax = partCountMin;

	// set the default map name if necessary
	if (!mapName)
		mapName = rigorous ? defMapNameRigorous : defMapName;

	// if nothing was requested to be computed, set the defaults
	if (!computeDelta && !computeDelta0 && !computeLambda &&
		!computeC && !computeLambda0)
	{
		computeLambda = 1;
		computeC = 1;
		computeLambda0 = 1;
	}

	// if the starting level exceeds the final level, make them equal
	if (finalLevel < startLevel)
		finalLevel = startLevel;

	// try running the main function and catch an error message if thrown
	try
	{
		// set an appropriate program time message
		program_time = "Aborted after:";
		program_time = 1;

		// run a test if this option was selected
		if (test)
		{
		}

		// run as a worker if requested to
		else if (work)
		{
			Worker w (controlNumber, false);
			w. Add (work);
			sout << "Running as a worker...\n";
			int result = w. Work ();
			if (result == mwOk)
				sout << "Work completed successfully.\n";
			else
				sout << "Could not work - probably "
					"an error occurred.\n";
		}

		// run as a coordinator or run the work locally otherwise
		else
		{
			// create a coordinator
			Coordinator c (controlNumber, filename, flushfile,
				paramMin, paramMax, intervals,
				deltaMin, deltaMax, resolution,
				lambdaMin, lambdaMax, mapName,
				partName, partCountMin, partCountMax,
				startLevel, finalLevel,
				computeDelta, computeDelta0,
				computeLambda, computeC, computeLambda0,
				rigorous, sparseGraph);
			c. Port (port);

			// run as a listening coordinator if requested to
			if (port)
			{
				sout << "Running as a coordinator...\n";
				int result = c. Coordinate ();
				if (result == mwOk)
					sout << "Completed successfully.\n";
				else
					sout << "Could not coordinate - "
						"an error occurred.\n";
			}

			// run the computations locally, otherwise
			else
			{
				Worker w (controlNumber, true);
				sout << "Running the computations...\n";
				int result = c. Coordinate (&w);
				if (result == mwOk)
					sout << "Completed successfully.\n";
				else
					sout << "Could not work - probably "
						"an error occurred.\n";
			}
		}

		// set an appropriate program time message
		program_time = "Total time used:";

		// finalize
		return 0;
	}
	catch (const char *msg)
	{
		sout << "ERROR: " << msg << '\n';
		return -1;
	}
	catch (const std::exception &e)
	{
		sout << "ERROR: " << e. what () << '\n';
		return -1;
	}
	catch (...)
	{
		sout << "ABORT: An unknown error occurred.\n";
		return -1;
	}
} /* main */

/// @}

