/// @addtogroup unifexp
/// @{

/////////////////////////////////////////////////////////////////////////////
///
/// @file partunif.h
///
/// This file contains the definition of a uniform partition type in which
/// the complement of the critical neighborhood is divided into intervals
/// of approximately the same length.
///
/// @author Pawel Pilarczyk
///
/////////////////////////////////////////////////////////////////////////////

// Copyright (C) 2007 by Pawel Pilarczyk.
//
// This file is part of my research program package.  This is free software;
// you can redistribute it and/or modify it under the terms of the GNU
// General Public License as published by the Free Software Foundation;
// either version 2 of the License, or (at your option) any later version.
//
// This software is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License along
// with this software; see the file "license.txt".  If not, write to the
// Free Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
// MA 02111-1307, USA.

// Started on August 21, 2007. Last revision: August 22, 2007.

#ifndef _partunif_h_
#define _partunif_h_

#include <vector>
#include <string>
#include "parttype.h"


namespace unifexp {

// --------------------------------------------------
// --------------- uniform partition ----------------
// --------------------------------------------------

/// A uniform partition type.
/// The complement of the critical neighborhood is divided into intervals
/// of approximately the same length.
template <class numType>
class partUniform: public partType<numType>
{
public:
	/// Returns the name of the object.
	std::string name () const;

	/// Creates a partition based on the given map, the requested
	/// number of elements in the partition, and the width of the
	/// critical neighborhood.
	void create (const mapType<numType> &theMap, int partCount,
		const numType &delta);

private:
	/// Fills part of the partition table between the given entries
	/// in a uniform way based on the values at these ends.
	void fillUniform (int first, int last);
	
}; /* class partUniform */

// --------------------------------------------------

template <class numType>
std::string partUniform<numType>::name () const
{
	return std::string ("uniform");
} /* partUniform::name */

template <class numType>
void partUniform<numType>::fillUniform (int first, int last)
{
	const numType &numFirst = (*this) [first];
	const numType &numLast = (*this) [last];
	const numType numDiff = (numLast - numFirst) / (last - first);
	for (int i = first + 1; i < last; ++ i)
		(*this) [i] = numFirst + (i - first) * numDiff;
	return;
} /* partUniform::fillUniform */

template <class numType>
void partUniform<numType>::create (const mapType<numType> &theMap,
	int partCount, const numType &delta)
{
	int nCrit = theMap. countCritical ();
	if (partCount < 2 * nCrit + 2)
		throw "Too small partition requested.";
	this -> allocate (partCount);
	(*this) [0] = theMap. leftBound ();
	(*this) [partCount] = theMap. rightBound ();
	numType width = (*this) [partCount] - (*this) [0] -
		2 * delta * nCrit;
	if (width <= 0)
		throw "Too wide critical neighborhood requested.";
	int prev = 0;
	for (int i = 0; i < nCrit; ++ i)
	{
		const numType crit = theMap. criticalPoint (i);
		const numType left = crit - delta;
		const numType right = crit + delta;
		if ((left <= (*this) [prev]) ||
			((*this) [partCount] <= right))
		{
			throw "Too large critical neighborhood requested.";
		}
		int n = prev + static_cast<int> (partCount / width *
			(left - (*this) [prev]));
		if (n <= prev)
			n = prev + 1;
		if (partCount <= n + 1)
			throw "Too few partition intervals requested.";
		this -> addCritical (n);
		(*this) [n] = left;
		fillUniform (prev, n);
		(*this) [n + 1] = right;
		prev = n + 1;
	}
	fillUniform (prev, partCount);
	return;
} /* partUniform::create */


} // namespace unifexp

#endif // _partunif_h_

/// @}

