/// @addtogroup unifexp
/// @{

/////////////////////////////////////////////////////////////////////////////
///
/// @file parttypes.h
///
/// This file contains the definition of a class which gathers all the
/// available partition types and allows to select a partition class object
/// by its name.
///
/// @author Pawel Pilarczyk
///
/////////////////////////////////////////////////////////////////////////////

// Copyright (C) 2007 by Pawel Pilarczyk.
//
// This file is part of my research program package.  This is free software;
// you can redistribute it and/or modify it under the terms of the GNU
// General Public License as published by the Free Software Foundation;
// either version 2 of the License, or (at your option) any later version.
//
// This software is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License along
// with this software; see the file "license.txt".  If not, write to the
// Free Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
// MA 02111-1307, USA.

// Started on August 22, 2007. Last revision: August 22, 2007.

#ifndef _parttypes_h_
#define _parttypes_h_

// some standard C++ headers
#include <vector>
#include <string>
#include <iostream>

// the abstract partition type header
#include "parttype.h"

// all the available partition type headers
#include "partunif.h"
#include "partsegm.h"
#include "partcrit.h"
#include "partderiv.h"

// ADD YOUR HEADERS HERE (and don't forget to add your objects below)



namespace unifexp {

// --------------------------------------------------
// ---------------- partition types -----------------
// --------------------------------------------------

/// This class gathers all the available partition types and allows
/// to select a partition class object by its name.
template <class numType>
class partTypes
{
public:
	/// The constructor.
	partTypes ();

	/// The destructor.
	~partTypes ();

	/// Retrieves a pointer to a partition object with the given name.
	/// Returns 0 if such an object cannot be found.
	partType<numType> *get (const std::string &name) const;

	/// Fills in a vector of text strings with the names
	/// of all the available partition objects.
	void getNames (std::vector<std::string> &names) const;

private:
	/// The copy constructor is not allowed.
	partTypes (const partTypes<numType> &);

	/// The assignment operator is not allowed.
	partTypes<numType> &operator = (const partTypes<numType> &);

	/// A vector of all the map objects to choose from.
	/// The objects must be created with the 'new' operator.
	/// They are automatically deallocated by the destructor.
	std::vector<partType<numType> *> objects;

}; /* class partTypes */

// --------------------------------------------------

template <class numType>
inline partTypes<numType>::partTypes ()
{
	// add objects of all map types
	objects. push_back (new partUniform<numType> ());
	objects. push_back (new partSegmented<numType> ());
	objects. push_back (new partCritical<numType> ());
	objects. push_back (new partDerivative<numType> ());

	// ADD YOUR OBJECTS HERE

	
	return;
} /* partTypes::partTypes */

template <class numType>
inline partTypes<numType>::~partTypes ()
{
	for (typename std::vector<partType<numType> *>::iterator iter =
		objects. begin (); iter != objects. end (); ++ iter)
	{
		delete *iter;
	}
	return;
} /* partTypes::~partTypes */

template <class numType>
inline partTypes<numType>::partTypes (const partTypes<numType> &)
{
	throw "Copy constructor not implemented for the partitions class.";
	return;
} /* partTypes::partTypes */

template <class numType>
inline partTypes<numType> &partTypes<numType>::operator =
	(const partTypes<numType> &)
{
	throw "Assignment operator not implemented "
		"for the partitions class.";
	return *this;
} /* partTypes::operator = */

template <class numType>
inline partType<numType> *partTypes<numType>::get (const std::string &name)
	const
{
	for (typename std::vector<partType<numType> *>::const_iterator
		iter = objects. begin (); iter != objects. end (); ++ iter)
	{
		if ((*iter) -> name () == name)
			return *iter;
	}
	return 0;
} /* partTypes::get */

template <class numType>
inline void partTypes<numType>::getNames (std::vector<std::string> &names)
	const
{
	for (typename std::vector<partType<numType> *>::const_iterator
		iter = objects. begin (); iter != objects. end (); ++ iter)
	{
		names. push_back ((*iter) -> name ());
	}
	return;
} /* partTypes::getNames */


} // namespace unifexp

#endif // _parttypes_h_

/// @}

