/// @addtogroup unifexp
/// @{

/////////////////////////////////////////////////////////////////////////////
///
/// @file mapunimi.h
///
/// This file contains the definition of the unimodal map "2|x|^gamma - a"
/// on [-1,1] with the usage of interval arithmetic
/// for rigorous computations.
///
/// @author Pawel Pilarczyk
///
/////////////////////////////////////////////////////////////////////////////

// Copyright (C) 2007 by Pawel Pilarczyk.
//
// This file is part of my research program package.  This is free software;
// you can redistribute it and/or modify it under the terms of the GNU
// General Public License as published by the Free Software Foundation;
// either version 2 of the License, or (at your option) any later version.
//
// This software is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License along
// with this software; see the file "license.txt".  If not, write to the
// Free Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
// MA 02111-1307, USA.

// Started on August 23, 2007. Last revision: February 3, 2013.

#ifndef _mapunimi_h_
#define _mapunimi_h_

#include <string>
#include <iostream>
#include <sstream>
#include "maptype.h"
#include "intvcapd.h"


namespace unifexp {

// --------------------------------------------------
// ---------------- the unimodal map ----------------
// --------------------------------------------------

/// This class defines the unimodal map "2 |x|^gamma - a" on [-1,1]
/// with the usage of interval arithmetic.
/// It is suitable for rigorous computations.
/// Recommended values of gamma are between 1 and 3.
/// Valid values of the parameter are between 0.5+ and 1 (optimal value: 1).
template <class numType>
class mapUnimodalIntv: public mapType<numType>
{
public:
	/// The constructor. Provide the numerator and the denominator
	/// for the exponent gamma to make sure its value is computed
	/// in a rigorous way as an interval.
	mapUnimodalIntv (const numType &_gamma1, const numType &_gamma2 = 1);

	/// Returns the name of the object.
	std::string name () const;

	/// Returns the number of critical points.
	int countCritical () const;

	/// Returns the subsequent critical points.
	numType criticalPoint (int n) const;

	/// Returns the left bound of the domain of the map.
	numType leftBound () const;

	/// Returns the right bound of the domain of the map.
	numType rightBound () const;

	/// Computes an enclosure of the image of the given interval.
	void image (const numType &x1, const numType &x2,
		numType &y1, numType &y2) const;

	/// Computes the minimal log of the derivative over those points
	/// in the interval [x1,x2] whose images may fall into [y1,y2]
	numType minLogDerivative (const numType &x1, const numType &x2,
		const numType &y1, const numType &y2) const;

private:
	/// The exponent of the map.
	IntervalType gamma;

	/// The cached value of log (2 gamma).
	IntervalType log2gamma;

	/// An auxiliary function for the computation of the root of degree
	/// gamma of the absolute value of a number.
	/// Returns the upper or lower bound for the result.
	numType gammaRoot (numType x, bool upperBound) const;

}; /* mapUnimodalIntv */

// --------------------------------------------------

template <class numType>
inline mapUnimodalIntv<numType>::mapUnimodalIntv (const numType &_gamma1,
	const numType &_gamma2):
	gamma (IntervalType (_gamma1, _gamma1) /
	IntervalType (_gamma2, _gamma2)),
	log2gamma (log (gamma * 2))
{
	return;
} /* mapUnimodalIntv::mapUnimodalIntv */

template <class numType>
inline std::string mapUnimodalIntv<numType>::name () const
{
	std::ostringstream nameStr;
	nameStr << "unimodal-intv-" << gamma. leftBound ();
	return nameStr. str ();
} /* mapUnimodalIntv::name */

template <class numType>
inline int mapUnimodalIntv<numType>::countCritical () const
{
	return 1;
} /* mapUnimodalIntv::countCritical */

template <class numType>
inline numType mapUnimodalIntv<numType>::criticalPoint (int n) const
{
	return 0.0;
} /* mapUnimodalIntv::criticalPoint */

template <class numType>
inline numType mapUnimodalIntv<numType>::leftBound () const
{
	return -1;
} /* mapUnimodalIntv::leftBound */

template <class numType>
inline numType mapUnimodalIntv<numType>::rightBound () const
{
	return 1;
} /* mapUnimodalIntv::rightBound */

template <class numType>
inline numType mapUnimodalIntv<numType>::gammaRoot (numType x,
	bool upperBound) const
{
	// return 0 if x is zero
	const numType zero (0);
	if (x == zero)
		return zero;

	// make x positive if it is negative
	if (x < zero)
		x = -x;

	// return the root of degree gamma of x
	if (upperBound)
	{
		return exp (log (IntervalType (x, x)) / gamma).
			rightBound ();
	}
	else
	{
		return exp (log (IntervalType (x, x)) / gamma).
			leftBound ();
	}

} /* mapUnimodalIntv::gammaRoot */

template <class numType>
inline void mapUnimodalIntv<numType>::image (const numType &x1,
	const numType &x2, numType &y1, numType &y2) const
{
	if (x2 < x1)
		throw "Image computation: Wrong interval for 'x'.";
	if ((x1 <= 0) && (0 <= x2))
	{
		const numType x = (-x1 < x2) ? x2 : -x1;
		y1 = -(this -> paramMax);
		y2 = (power (IntervalType (x, x), gamma) * 2 -
			this -> paramMin). rightBound ();
	}
	else
	{
		IntervalType img = power ((x2 < 0) ?
			IntervalType (-x2, -x1) :
			IntervalType (x1, x2), gamma) * 2 -
			IntervalType (this -> paramMin, this -> paramMax);
		y1 = img. leftBound ();
		y2 = img. rightBound ();
	}
	resetRounding ();
	return;
} /* mapUnimodalIntv::image */

template <class numType>
inline numType mapUnimodalIntv<numType>::minLogDerivative (const numType &x1,
	const numType &x2, const numType &y1, const numType &y2) const
{
	// make sure the input data is correct
	if (x2 < x1)
		throw "MinLogDerivative: Wrong interval for 'x'.";
	if (y2 < y1)
		throw "MinLogDerivative: Wrong interval for 'y'.";
	if (((x1 < 0) || (x1 == 0)) && ((x2 == 0) || (0 < x2)))
		throw "MinLogDerivative: The interval contains zero.";

	// compute the positive preimage of the interval [y1,y2],
	// intersect the computed preimage with the interval [x1,x2],
	// and return the log of the derivative at the endpoint at which
	// the derivative is minimal (right for gamma < 1, left otherwise)
	numType result;
	if (gamma. rightBound () < 1)
	{
		const numType sum = ((IntervalType (y2, y2) +
			this -> paramMax) / 2). rightBound ();
		const numType preImg = (0 < sum) ?
			gammaRoot (sum, true) : 0;
		const numType x = (0 < x1) ? x2 : -x1;
		const numType endPoint = (x < preImg) ? x : preImg;
		result = (log2gamma + (gamma - 1) *
			log (IntervalType (endPoint, endPoint))).
			leftBound ();
	}
	else
	{
		const numType sum = ((IntervalType (y1, y1) +
			this -> paramMin) / 2). leftBound ();
		if ((sum < 0) || (sum == 0))
			throw "MinLogDerivative: Zero in the preimage.";
		const numType preImg = (0 < sum) ?
			gammaRoot (sum, false) : 0;
		const numType x = (0 < x1) ? x1 : -x2;
		const numType endPoint = (x < preImg) ? preImg : x;
		result = (log2gamma + (gamma - 1) *
			log (IntervalType (endPoint, endPoint))).
			leftBound ();
	}
	resetRounding ();
	return result;
} /* mapUnimodalIntv::minLogDerivative */


} // namespace unifexp

#endif // _mapunimi_h_

/// @}

