/// @addtogroup unifexp
/// @{

/////////////////////////////////////////////////////////////////////////////
///
/// @file mapunim.h
///
/// This file contains the definition of the unimodal map "2|x|^gamma - a"
/// on [-1,1] without using interval arithmetic
/// for non-rigorous computations.
///
/// @author Pawel Pilarczyk
///
/////////////////////////////////////////////////////////////////////////////

// Copyright (C) 2007 by Pawel Pilarczyk.
//
// This file is part of my research program package.  This is free software;
// you can redistribute it and/or modify it under the terms of the GNU
// General Public License as published by the Free Software Foundation;
// either version 2 of the License, or (at your option) any later version.
//
// This software is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License along
// with this software; see the file "license.txt".  If not, write to the
// Free Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
// MA 02111-1307, USA.

// Started on August 23, 2007. Last revision: August 23, 2007.

#ifndef _mapunim_h_
#define _mapunim_h_

#include <string>
#include <iostream>
#include <sstream>
#include "maptype.h"


namespace unifexp {

// --------------------------------------------------
// ---------------- the unimodal map ----------------
// --------------------------------------------------

/// This class defines the unimodal map "2 |x|^gamma - a" on [-1,1]
/// without using interval arithmetic.
/// It is suitable for non-rigorous computations.
/// See the class "mapUnimodalIntv" for a rigorous version which does use
/// interval arithmetic.
/// Recommended values of gamma are between 1 and 3.
/// Valid values of the parameter are between 0.5+ and 1 (optimal value: 1).
template <class numType>
class mapUnimodal: public mapType<numType>
{
public:
	/// The constructor.
	mapUnimodal (const numType &_gamma);

	/// Returns the name of the object.
	std::string name () const;

	/// Returns the number of critical points.
	int countCritical () const;

	/// Returns the subsequent critical points.
	numType criticalPoint (int n) const;

	/// Returns the left bound of the domain of the map.
	numType leftBound () const;

	/// Returns the right bound of the domain of the map.
	numType rightBound () const;

	/// Computes an enclosure of the image of the given interval.
	void image (const numType &x1, const numType &x2,
		numType &y1, numType &y2) const;

	/// Computes the minimal log of the derivative over those points
	/// in the interval [x1,x2] whose images may fall into [y1,y2]
	numType minLogDerivative (const numType &x1, const numType &x2,
		const numType &y1, const numType &y2) const;

private:
	/// The exponent of the map.
	numType gamma;

	/// An auxiliary function for the computation of the absolute value
	/// of a number raised to the power gamma.
	numType gammaPower (numType x) const;

	/// An auxiliary function for the computation of the absolute value
	/// of a number raised to the power gamma - 1.
	numType gammaPower1 (numType x) const;

	/// An auxiliary function for the computation of the root of degree
	/// gamma of the absolute value of a number.
	numType gammaRoot (numType x) const;

}; /* mapUnimodal */

// --------------------------------------------------

template <class numType>
inline mapUnimodal<numType>::mapUnimodal (const numType &_gamma):
	gamma (_gamma)
{
	return;
} /* mapUnimodal::mapUnimodal */

template <class numType>
inline std::string mapUnimodal<numType>::name () const
{
	std::ostringstream nameStr;
	nameStr << "unimodal-" << gamma;
	return nameStr. str ();
} /* mapUnimodal::name */

template <class numType>
inline int mapUnimodal<numType>::countCritical () const
{
	return 1;
} /* mapUnimodal::countCritical */

template <class numType>
inline numType mapUnimodal<numType>::criticalPoint (int n) const
{
	return 0.0;
} /* mapUnimodal::criticalPoint */

template <class numType>
inline numType mapUnimodal<numType>::leftBound () const
{
	return -1;
} /* mapUnimodal::leftBound */

template <class numType>
inline numType mapUnimodal<numType>::rightBound () const
{
	return 1;
} /* mapUnimodal::rightBound */

template <class numType>
inline numType mapUnimodal<numType>::gammaPower (numType x) const
{
	// return x if raising to power 1
//	const numType one (1);
//	if (gamma == one)
//		return x;

	// return 0 if x is zero
	const numType zero (0);
	if (x == zero)
		return zero;

	// make x positive if it is negative
	if (x < zero)
		x = -x;

	// return x to the power gamma
	return exp (log (x) * gamma);

} /* mapUnimodal::gammaPower */

template <class numType>
inline numType mapUnimodal<numType>::gammaPower1 (numType x) const
{
	// return x if raising to power 1
//	const numType two (2);
//	if (gamma == two)
//		return x;

	// return 1 if raising to power 0
//	const numType one (1);
//	if (gamma == one)
//		return 1;

	// return 0 if x is zero
	const numType zero (0);
	if (x == zero)
		return zero;

	// make x positive if it is negative
	if (x < zero)
		x = -x;

	// return x to the power gamma
	return exp (log (x) * (gamma - 1));

} /* mapUnimodal::gammaPower1 */

template <class numType>
inline numType mapUnimodal<numType>::gammaRoot (numType x) const
{
	// return x if computing the root of degree 1
//	const numType one (1);
//	if (gamma == one)
//		return x;

	// return 0 if x is zero
	const numType zero (0);
	if (x == zero)
		return zero;

	// make x positive if it is negative
	if (x < zero)
		x = -x;

	// return the root of degree gamma of x
	return exp (log (x) / gamma);

} /* mapUnimodal::gammaRoot */

template <class numType>
inline void mapUnimodal<numType>::image (const numType &x1,
	const numType &x2, numType &y1, numType &y2) const
{
	if (x2 < x1)
		throw "Image computation: Wrong interval for 'x'.";
	if (x1 > 0)
	{
		y1 = 2 * gammaPower (x1) - this -> paramMax;
		y2 = 2 * gammaPower (x2) - this -> paramMin;
	}
	else if (x2 < 0)
	{
		y1 = 2 * gammaPower (-x2) - this -> paramMax;
		y2 = 2 * gammaPower (-x1) - this -> paramMin;
	}
	else
	{
		y2 = 2 * gammaPower ((-x1 < x2) ? x2 : -x1) -
			this -> paramMin;
		y1 = -this -> paramMax;
	}
	return;
} /* mapUnimodal::image */

template <class numType>
inline numType mapUnimodal<numType>::minLogDerivative (const numType &x1,
	const numType &x2, const numType &y1, const numType &y2) const
{
	// make sure the input data is correct
	if (x2 < x1)
		throw "MinLogDerivative: Wrong interval for 'x'.";
	if (y2 < y1)
		throw "MinLogDerivative: Wrong interval for 'y'.";
	if ((x1 <= 0) && (0 <= x2))
		throw "MinLogDerivative: The interval contains zero.";

	// compute the positive preimage of the interval [y1,y2],
	// intersect the computed preimage with the interval [x1,x2],
	// and return the log of the derivative at the endpoint at which
	// the derivative is minimal (right for gamma < 1, left otherwise)
	if (gamma < 1)
	{
		const numType sum = ((y2 + this -> paramMax) / 2);
		const numType preImg = (0 < sum) ? gammaRoot (sum) : 0;
		const numType x = (0 < x1) ? x2 : -x1;
		const numType endPoint = (x < preImg) ? x : preImg;
		return log (2 * gamma) + (gamma - 1) * log (endPoint);
	}
	else
	{
		const numType sum = ((y1 + this -> paramMin) / 2);
		if ((sum < 0) || (sum == 0))
			throw "MinLogDerivative: Zero in the preimage.";
		const numType preImg = (0 < sum) ? gammaRoot (sum) : 0;
		const numType x = (0 < x1) ? x1 : -x2;
		const numType endPoint = (x < preImg) ? preImg : x;
		return log (2 * gamma) + (gamma - 1) * log (endPoint);
	}
} /* mapUnimodal::minLogDerivative */


} // namespace unifexp

#endif // _mapunim_h_

/// @}

