/// @addtogroup unifexp
/// @{

/////////////////////////////////////////////////////////////////////////////
///
/// @file maptypes.h
///
/// This file contains the definition of a class which gathers all the
/// available map types and allows to select a map class object by its name.
///
/// @author Pawel Pilarczyk
///
/////////////////////////////////////////////////////////////////////////////

// Copyright (C) 2007 by Pawel Pilarczyk.
//
// This file is part of my research program package.  This is free software;
// you can redistribute it and/or modify it under the terms of the GNU
// General Public License as published by the Free Software Foundation;
// either version 2 of the License, or (at your option) any later version.
//
// This software is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License along
// with this software; see the file "license.txt".  If not, write to the
// Free Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
// MA 02111-1307, USA.

// Started on August 22, 2007. Last revision: August 23, 2007.

#ifndef _maptypes_h_
#define _maptypes_h_

// some standard C++ headers
#include <vector>
#include <string>
#include <iostream>

// the abstract map type header
#include "maptype.h"

// all the available map type headers
#include "mapquad.h"
#include "mapquadi.h"
#include "mapunim.h"
#include "mapunimi.h"
#include "mapcub.h"
#include "mapcubi.h"

// ADD YOUR HEADERS HERE (and don't forget to add your objects below)



namespace unifexp {

// --------------------------------------------------
// ------------------- map types --------------------
// --------------------------------------------------

/// This class gathers all the available map types and allows
/// to select a map class object by its name.
template <class numType>
class mapTypes
{
public:
	/// The constructor.
	mapTypes ();

	/// The destructor.
	~mapTypes ();

	/// Retrieves a pointer to a map object with the given name.
	/// Returns 0 if such an object cannot be found.
	mapType<numType> *get (const std::string &name) const;

	/// Fills in a vector of text strings with the names
	/// of all the available map objects.
	void getNames (std::vector<std::string> &names) const;

private:
	/// The copy constructor is not allowed.
	mapTypes (const mapTypes<numType> &);

	/// The assignment operator is not allowed.
	mapTypes<numType> &operator = (const mapTypes<numType> &);

	/// A vector of all the map objects to choose from.
	/// The objects must be created with the 'new' operator.
	/// They are automatically deallocated by the destructor.
	std::vector<mapType<numType> *> objects;

}; /* class mapTypes */

// --------------------------------------------------

template <class numType>
inline mapTypes<numType>::mapTypes ()
{
	// add objects of all map types
	objects. push_back (new mapQuadr<numType> ());
	objects. push_back (new mapQuadrIntv<numType> ());
	objects. push_back (new mapUnimodal<numType> (1.0));
	objects. push_back (new mapUnimodal<numType> (3.0 / 2));
	objects. push_back (new mapUnimodal<numType> (2.0));
	objects. push_back (new mapUnimodal<numType> (5.0 / 2));
	objects. push_back (new mapUnimodalIntv<numType> (1.0));
	objects. push_back (new mapUnimodalIntv<numType> (3.0, 2.0));
	objects. push_back (new mapUnimodalIntv<numType> (2.0));
	objects. push_back (new mapUnimodalIntv<numType> (5.0, 2.0));
	objects. push_back (new mapCubic<numType> ());
	objects. push_back (new mapCubicIntv<numType> ());

	// ADD YOUR OBJECTS HERE

	
	return;
} /* mapTypes::mapTypes */

template <class numType>
inline mapTypes<numType>::~mapTypes ()
{
	for (typename std::vector<mapType<numType> *>::iterator iter =
		objects. begin (); iter != objects. end (); ++ iter)
	{
		delete *iter;
	}
	return;
} /* mapTypes::~mapTypes */

template <class numType>
inline mapTypes<numType>::mapTypes (const mapTypes<numType> &)
{
	throw "Copy constructor not implemented for the map types class.";
	return;
} /* mapTypes::mapTypes */

template <class numType>
inline mapTypes<numType> &mapTypes<numType>::operator =
	(const mapTypes<numType> &)
{
	throw "Assignment operator not implemented for the map types class.";
	return *this;
} /* mapTypes::operator = */

template <class numType>
inline mapType<numType> *mapTypes<numType>::get (const std::string &name)
	const
{
	for (typename std::vector<mapType<numType> *>::const_iterator
		iter = objects. begin (); iter != objects. end (); ++ iter)
	{
		if ((*iter) -> name () == name)
			return *iter;
	}
	return 0;
} /* mapTypes::get */

template <class numType>
inline void mapTypes<numType>::getNames (std::vector<std::string> &names)
	const
{
	for (typename std::vector<mapType<numType> *>::const_iterator
		iter = objects. begin (); iter != objects. end (); ++ iter)
	{
		names. push_back ((*iter) -> name ());
	}
	return;
} /* mapTypes::getNames */


} // namespace unifexp

#endif // _maptypes_h_

/// @}

