/// @addtogroup unifexp
/// @{

/////////////////////////////////////////////////////////////////////////////
///
/// @file maptype.h
///
/// This file contains the definition of an abstract map type.
///
/// @author Pawel Pilarczyk
///
/////////////////////////////////////////////////////////////////////////////

// Copyright (C) 2007 by Pawel Pilarczyk.
//
// This file is part of my research program package.  This is free software;
// you can redistribute it and/or modify it under the terms of the GNU
// General Public License as published by the Free Software Foundation;
// either version 2 of the License, or (at your option) any later version.
//
// This software is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License along
// with this software; see the file "license.txt".  If not, write to the
// Free Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
// MA 02111-1307, USA.

// Started on March 8, 2007. Last revision: May 1, 2007.

#ifndef _maptype_h_
#define _maptype_h_

#include <vector>
#include <string>
#include <iostream>


namespace unifexp {

// --------------------------------------------------
// -------------------- map type --------------------
// --------------------------------------------------

/// This is an abstract map type.
/// Each specific map class must inherit from this class.
template <class numType>
class mapType
{
public:
	/// The constructor.
	mapType ();

	/// The virtual destructor.
	virtual ~mapType ();

	/// Returns the name of the object.
	virtual std::string name () const = 0;

	/// Sets the parameters of the map.
	void setParam (const numType &_paramMin, const numType &_paramMax);

	/// Returns the number of critical points.
	virtual int countCritical () const = 0;

	/// Returns the subsequent critical points.
	virtual numType criticalPoint (int n) const = 0;

	/// Returns the left bound of the domain of the map.
	virtual numType leftBound () const = 0;

	/// Returns the right bound of the domain of the map.
	virtual numType rightBound () const = 0;

	/// Computes an enclosure of the image of the given interval.
	virtual void image (const numType &x1, const numType &x2,
		numType &y1, numType &y2) const = 0;

	/// Computes the minimal log of the derivative over those points
	/// in the interval [x1,x2] whose images may fall into [y1,y2]
	virtual numType minLogDerivative
		(const numType &x1, const numType &x2,
		const numType &y1, const numType &y2) const = 0;

protected:
	/// The minimal parameter of the map.
	numType paramMin;

	/// The maximal parameter of the map.
	numType paramMax;

private:
	/// Copy constructor not allowed.
	mapType (const mapType<numType> &);

	/// Assignment operator not allowed.
	mapType<numType> &operator = (const mapType<numType> &);

}; /* class mapType */

// --------------------------------------------------

template <class numType>
inline mapType<numType>::mapType ()
{
	return;
} /* mapType::mapType */

template <class numType>
inline mapType<numType>::~mapType ()
{
	return;
} /* mapType::~mapType */

template <class numType>
inline mapType<numType>::mapType (const mapType<numType> &)
{
	throw "Copy constructor not implemented for map types.";
	return;
} /* mapType::mapType */

template <class numType>
inline mapType<numType> &mapType<numType>::operator =
	(const mapType<numType> &)
{
	throw "Assignment operator not implemented for map types.";
	return *this;
} /* mapType::operator = */

template <class numType>
inline void mapType<numType>::setParam (const numType &_paramMin,
	const numType &_paramMax)
{
	paramMin = _paramMin;
	paramMax = _paramMax;
	if (_paramMin > _paramMax)
	{
		paramMin = _paramMax;
		paramMax = _paramMin;
	}
	return;
} /* mapType::setParam */


} // namespace unifexp

#endif // _maptype_h_

/// @}

