/// @addtogroup unifexp
/// @{

/////////////////////////////////////////////////////////////////////////////
///
/// @file mapquadi.h
///
/// This file contains the definition of the quadratic map f(x)=x^2-a
/// with the use of interval arithmetic for rigorous computations.
///
/// @author Pawel Pilarczyk
///
/////////////////////////////////////////////////////////////////////////////

// Copyright (C) 2007 by Pawel Pilarczyk.
//
// This file is part of my research program package.  This is free software;
// you can redistribute it and/or modify it under the terms of the GNU
// General Public License as published by the Free Software Foundation;
// either version 2 of the License, or (at your option) any later version.
//
// This software is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License along
// with this software; see the file "license.txt".  If not, write to the
// Free Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
// MA 02111-1307, USA.

// Started on March 8, 2007. Last revision: February 3, 2013.

#ifndef _mapquadi_h_
#define _mapquadi_h_

#include <string>
#include <iostream>
#include "maptype.h"
#include "intvcapd.h"


namespace unifexp {

// --------------------------------------------------
// --------------- the quadratic map ----------------
// ------------ with interval arithmetic ------------
// --------------------------------------------------

/// This class defines the quadratic map with the use of interval arithmetic.
/// It is suitable for doing rigorous computations.
template <class numType>
class mapQuadrIntv: public mapType<numType>
{
public:
	/// The constructor.
	mapQuadrIntv ();

	/// Returns the name of the object.
	std::string name () const;

	/// Returns the number of critical points.
	int countCritical () const;

	/// Returns the subsequent critical points.
	numType criticalPoint (int n) const;

	/// Returns the left bound of the domain of the map.
	numType leftBound () const;

	/// Returns the right bound of the domain of the map.
	numType rightBound () const;

	/// Computes an enclosure of the image of the given interval.
	void image (const numType &x1, const numType &x2,
		numType &y1, numType &y2) const;

	/// Computes the minimal log of the derivative over those points
	/// in the interval [x1,x2] whose images may fall into [y1,y2]
	numType minLogDerivative (const numType &x1, const numType &x2,
		const numType &y1, const numType &y2) const;

}; /* mapQuadrIntv */

// --------------------------------------------------

template <class numType>
mapQuadrIntv<numType>::mapQuadrIntv ()
{
	return;
} /* mapQuadrIntv::mapQuadrIntv */

template <class numType>
std::string mapQuadrIntv<numType>::name () const
{
	return std::string ("quadratic-intv");
} /* mapQuadrIntv::name */

template <class numType>
inline int mapQuadrIntv<numType>::countCritical () const
{
	return 1;
} /* mapQuadrIntv::countCritical */

template <class numType>
inline numType mapQuadrIntv<numType>::criticalPoint (int n) const
{
	return 0.0;
} /* mapQuadrIntv::criticalPoint */

template <class numType>
inline numType mapQuadrIntv<numType>::leftBound () const
{
	return -2;
//	return -this -> paramMax;
} /* mapQuadrIntv::leftBound */

template <class numType>
inline numType mapQuadrIntv<numType>::rightBound () const
{
	return 2;
//	numType result = (power (IntervalType (this -> paramMax,
//		this -> paramMax), 2) - this -> paramMin). rightBound ();
//	resetRounding ();
//	return result;
} /* mapQuadrIntv::rightBound */

template <class numType>
void mapQuadrIntv<numType>::image (const numType &x1, const numType &x2,
	numType &y1, numType &y2) const
{
	if (x2 < x1)
		throw "Image computation: Wrong interval for 'x'.";
	IntervalType img = power (IntervalType (x1, x2), 2) -
		IntervalType (this -> paramMin, this -> paramMax);
	y1 = img. leftBound ();
	y2 = img. rightBound ();
	resetRounding ();
	return;
} /* mapQuadrIntv::image */

template <class numType>
numType mapQuadrIntv<numType>::minLogDerivative (const numType &x1,
	const numType &x2, const numType &y1, const numType &y2) const
{
	// make sure the input data is correct
	if (x2 < x1)
		throw "MinLogDerivative: Wrong interval for 'x'.";
	if (y2 < y1)
		throw "MinLogDerivative: Wrong interval for 'y'.";

	// compute the positive preimage of the interval [y1,y2],
	// intersect the computed preimage with the interval [x1,x2],
	// and return the log of the derivative at the minimal endpoint
	const numType sum1 = (IntervalType (y1, y1) + this -> paramMin).
		leftBound ();
	const numType left1 = (0 < sum1) ?
		(sqrt (IntervalType (sum1, sum1))). leftBound () : 0;
	numType result;
	if (0 < x1)
	{
		const numType left = (left1 > x1) ? left1 : x1;
		result = (log (2 * IntervalType (left, left))). leftBound ();
	}
	else if (x2 < 0)
	{
		const numType left = (left1 > -x2) ? left1 : -x2;
		result = (log (2 * IntervalType (left, left))). leftBound ();
	}
	else
	{
		throw "Log of interval containing zero.";
		result = 0;
	}
	resetRounding ();
	return result;
} /* mapQuadrIntv::minLogDerivative */


} // namespace unifexp

#endif // _mapquadi_h_

/// @}

