/// @addtogroup unifexp
/// @{

/////////////////////////////////////////////////////////////////////////////
///
/// @file mapquad.h
///
/// This file contains the definition of the quadratic map f(x)=x^2-a
/// without using interval arithmetic for non-rigorous computations.
///
/// @author Pawel Pilarczyk
///
/////////////////////////////////////////////////////////////////////////////

// Copyright (C) 2007 by Pawel Pilarczyk.
//
// This file is part of my research program package.  This is free software;
// you can redistribute it and/or modify it under the terms of the GNU
// General Public License as published by the Free Software Foundation;
// either version 2 of the License, or (at your option) any later version.
//
// This software is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License along
// with this software; see the file "license.txt".  If not, write to the
// Free Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
// MA 02111-1307, USA.

// Started on March 8, 2007. Last revision: May 1, 2007.

#ifndef _mapquadr_h_
#define _mapquadr_h_

#include <string>
#include <iostream>
#include "maptype.h"
#include "chomp/system/textfile.h" // for debugging only


namespace unifexp {

// --------------------------------------------------
// --------------- the quadratic map ----------------
// --------------------------------------------------

/// This class defines the quadratic map f(x)=x^2-a
/// without using interval arithmetic.
/// It is suitable for non-rigorous computations.
/// See the class "mapQuadrIntv" for a rigorous version which does use
/// interval arithmetic.
/// Valid values of the parameter are between 1+ and 2 (optimal value: 2).
template <class numType>
class mapQuadr: public mapType<numType>
{
public:
	/// The constructor.
	mapQuadr ();

	/// Returns the name of the object.
	std::string name () const;

	/// Returns the number of critical points.
	int countCritical () const;

	/// Returns the subsequent critical points.
	numType criticalPoint (int n) const;

	/// Returns the left bound of the domain of the map.
	numType leftBound () const;

	/// Returns the right bound of the domain of the map.
	numType rightBound () const;

	/// Computes an enclosure of the image of the given interval.
	void image (const numType &x1, const numType &x2,
		numType &y1, numType &y2) const;

	/// Computes the minimal log of the derivative over those points
	/// in the interval [x1,x2] whose images may fall into [y1,y2]
	numType minLogDerivative (const numType &x1, const numType &x2,
		const numType &y1, const numType &y2) const;

}; /* mapQuadr */

// --------------------------------------------------

template <class numType>
inline mapQuadr<numType>::mapQuadr ()
{
	return;
} /* mapQuadr::mapQuadr */

template <class numType>
inline std::string mapQuadr<numType>::name () const
{
	return std::string ("quadratic");
} /* mapQuadr::name */

template <class numType>
inline int mapQuadr<numType>::countCritical () const
{
	return 1;
} /* mapQuadr::countCritical */

template <class numType>
inline numType mapQuadr<numType>::criticalPoint (int n) const
{
	return 0.0;
} /* mapQuadr::criticalPoint */

template <class numType>
inline numType mapQuadr<numType>::leftBound () const
{
	return -2;
//	return -this -> paramMax;
} /* mapQuadr::leftBound */

template <class numType>
inline numType mapQuadr<numType>::rightBound () const
{
	return 2;
//	return this -> paramMax * this -> paramMax - this -> paramMin;
} /* mapQuadr::rightBound */

template <class numType>
inline void mapQuadr<numType>::image (const numType &x1, const numType &x2,
	numType &y1, numType &y2) const
{
	if (x2 < x1)
		throw "Image computation: Wrong interval for 'x'.";
	if (x1 > 0)
	{
		y1 = x1 * x1 - this -> paramMax;
		y2 = x2 * x2 - this -> paramMin;
	}
	else if (x2 < 0)
	{
		y1 = x2 * x2 - this -> paramMax;
		y2 = x1 * x1 - this -> paramMin;
	}
	else
	{
		y2 = ((-x1 < x2) ? (x2 * x2) : (x1 * x1)) - this -> paramMin;
		y1 = -this -> paramMax;
	}
	return;
} /* mapQuadr::image */

template <class numType>
inline numType mapQuadr<numType>::minLogDerivative (const numType &x1,
	const numType &x2, const numType &y1, const numType &y2) const
{
	// make sure the input data is correct
	if (x2 < x1)
		throw "MinLogDerivative: Wrong interval for 'x'.";
	if (y2 < y1)
		throw "MinLogDerivative: Wrong interval for 'y'.";

	// compute the positive preimage of the interval [y1,y2],
	// intersect the computed preimage with the interval [x1,x2],
	// and return the log of the derivative at the minimal endpoint
	const numType sum1 = y1 + this -> paramMin;
	numType left = (0 < sum1) ? sqrt (sum1) : 0;
	if (0 < x1)
	{
		return log (2 * ((left > x1) ? left : x1));
	}
	else if (x2 < 0)
	{
		return log (2 * ((left > -x2) ? left : -x2));
	}
	else
	{
		chomp::homology::sbug << "x1 = " << x1 << ", x2 = " << x2 << "\n";
		throw "Log* of interval containing zero.";
		return 0;
	}
} /* mapQuadr::minLogDerivative */


} // namespace unifexp

#endif // _mapquadr_h_

/// @}

