/// @addtogroup unifexp
/// @{

/////////////////////////////////////////////////////////////////////////////
///
/// @file mapcub.h
///
/// This file contains the definition of the cubic map -x^3 + 3x - a
/// on [-4,4] without using interval arithmetic
/// for non-rigorous computations.
///
/// @author Pawel Pilarczyk
///
/////////////////////////////////////////////////////////////////////////////

// Copyright (C) 2007 by Pawel Pilarczyk.
//
// This file is part of my research program package.  This is free software;
// you can redistribute it and/or modify it under the terms of the GNU
// General Public License as published by the Free Software Foundation;
// either version 2 of the License, or (at your option) any later version.
//
// This software is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License along
// with this software; see the file "license.txt".  If not, write to the
// Free Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
// MA 02111-1307, USA.

// Started on August 24, 2007. Last revision: August 24, 2007.

#ifndef _mapcub_h_
#define _mapcub_h_

#include <string>
#include <iostream>
#include "maptype.h"


namespace unifexp {

// --------------------------------------------------
// ----------------- the cubic map ------------------
// --------------------------------------------------

/// This class defines the cubic map -x^3 + 3x - a on [-4,4]
/// without using interval arithmetic.
/// It is suitable for non-rigorous computations.
/// See the class "mapCubicIntv" for a rigorous version which does use
/// interval arithmetic.
/// Good values of the parameter are between 0 and 2,
/// more or less (the optimal value is 2, I think).
template <class numType>
class mapCubic: public mapType<numType>
{
public:
	/// The constructor.
	mapCubic ();

	/// Returns the name of the object.
	std::string name () const;

	/// Returns the number of critical points.
	int countCritical () const;

	/// Returns the subsequent critical points.
	numType criticalPoint (int n) const;

	/// Returns the left bound of the domain of the map.
	numType leftBound () const;

	/// Returns the right bound of the domain of the map.
	numType rightBound () const;

	/// Computes an enclosure of the image of the given interval.
	void image (const numType &x1, const numType &x2,
		numType &y1, numType &y2) const;

	/// Computes the minimal log of the derivative over those points
	/// in the interval [x1,x2] whose images may fall into [y1,y2]
	numType minLogDerivative (const numType &x1, const numType &x2,
		const numType &y1, const numType &y2) const;

}; /* mapCubic */

// --------------------------------------------------

template <class numType>
inline mapCubic<numType>::mapCubic ()
{
	return;
} /* mapCubic::mapCubic */

template <class numType>
inline std::string mapCubic<numType>::name () const
{
	return std::string ("cubic");
} /* mapCubic::name */

template <class numType>
inline int mapCubic<numType>::countCritical () const
{
	return 2;
} /* mapCubic::countCritical */

template <class numType>
inline numType mapCubic<numType>::criticalPoint (int n) const
{
	return n ? 1 : -1;
} /* mapCubic::criticalPoint */

template <class numType>
inline numType mapCubic<numType>::leftBound () const
{
	return -4;
} /* mapCubic::leftBound */

template <class numType>
inline numType mapCubic<numType>::rightBound () const
{
	return 4;
} /* mapCubic::rightBound */

template <class numType>
inline void mapCubic<numType>::image (const numType &x1, const numType &x2,
	numType &y1, numType &y2) const
{
	if (x2 < x1)
		throw "Image computation: Wrong interval for 'x'.";
	if ((x2 <= -1) || (1 <= x1))
	{
		y1 = -(x2 * x2 - 3) * x2 - this -> paramMax;
		y2 = -(x1 * x1 - 3) * x1 - this -> paramMin;
	}
	else if ((-1 <= x1) && (x2 <= 1))
	{
		y1 = -(x1 * x1 - 3) * x1 - this -> paramMax;
		y2 = -(x2 * x2 - 3) * x2 - this -> paramMin;
	}
	else if (x2 <= 1)
	{
		const numType y3 = -(x1 * x1 - 3) * x1;
		const numType y4 = -(x2 * x2 - 3) * x2;
		y1 = -2 - this -> paramMax;
		y2 = ((y3 < y4) ? y4 : y3) - this -> paramMin;
	}
	else if (-1 <= x1)
	{
		const numType y3 = -(x1 * x1 - 3) * x1;
		const numType y4 = -(x2 * x2 - 3) * x2;
		y1 = ((y3 < y4) ? y3 : y4) - this -> paramMax;
		y2 = 4 - this -> paramMin;
	}
	else
		throw "Cubic map: Too large domain interval.";
	return;
} /* mapCubic::image */

template <class numType>
inline numType mapCubic<numType>::minLogDerivative (const numType &x1,
	const numType &x2, const numType &y1, const numType &y2) const
{
	// make sure the input data is correct
	if (x2 < x1)
		throw "MinLogDerivative: Wrong interval for 'x'.";
	if (y2 < y1)
		throw "MinLogDerivative: Wrong interval for 'y'.";

	// NOTE: Since it is not so easy to compute the preimage of the
	// interval [y1,y2] with respect to the cubic polynomial map,
	// the lower bound for the log of the derivative is computed
	// solely based on the interval [x1,x2]
	if (x2 < -1)
	{
		return log (3 * x2 * x2 + 3);
	}
	else if (1 < x1)
	{
		return log (3 * x1 * x1 + 3);
	}
	else if ((-1 < x1) && (x2 < 1))
	{
		if (0 < x1)
			return log (-3 * x2 * x2 + 3);
		else if (x2 < 0)
			return log (-3 * x1 * x1 + 3);
		else
		{
			return log (-3 * ((x1 < x2) ?
				(x2 * x2) : (x1 * x1)) + 3);
		}
	}
	else
	{
		throw "Log of interval containing zero.";
		return 0;
	}
} /* mapCubic::minLogDerivative */


} // namespace unifexp

#endif // _mapcub_h_

/// @}

