/// @addtogroup unifexp
/// @{

/////////////////////////////////////////////////////////////////////////////
///
/// @file coord.h
///
/// This file contains the definition of the class "coordinator"
/// which coordinates series of computations.
///
/// @author Pawel Pilarczyk
///
/////////////////////////////////////////////////////////////////////////////

// Copyright (C) 2007 by Pawel Pilarczyk.
//
// This file is part of my research program package.  This is free software;
// you can redistribute it and/or modify it under the terms of the GNU
// General Public License as published by the Free Software Foundation;
// either version 2 of the License, or (at your option) any later version.
//
// This software is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License along
// with this software; see the file "license.txt".  If not, write to the
// Free Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
// MA 02111-1307, USA.

// Started on March 8, 2007. Last revision: June 1, 2008.

#ifndef _coord_h_
#define _coord_h_

#include <iostream>
#include <fstream>
#include <iomanip>
#include <vector>

#include "chomp/system/config.h"
#include "chomp/system/textfile.h"
#include "chomp/system/timeused.h"
#include "chomp/multiwork/mw.h"

#include "bitvect.h"


namespace unifexp {

// --------------------------------------------------
// ------------- the coordinator class --------------
// --------------------------------------------------

/// An object of this class coordinates a series of computations
/// for the Uniform Expansion program.
/// This class inherits from the mwCoordinator class
/// to allow network distribution of the work.
class Coordinator: public chomp::multiwork::mwCoordinator
{
public:
	/// The complete constructor of a coordinator.
	Coordinator (int _controlNumber,
		const char *_filename, bool _flushfile,
		double _paramMin, double _paramMax, bool _intervals,
		double _deltaMin, double _deltaMax, double _resolution,
		double _lambdaMin, double _lambdaMax, const char *_mapName,
		const char *_partName, int _partCountMin, int _partCountMax,
		int _startLevel, int _finalLevel,
		int _computeDelta, int _computeDelta0,
		int _computeLambda, int _computeC, int _computeLambda0,
		int _rigorous, int _sparseGraph);

	/// The destructor of the coordinator.
	~Coordinator ();

private:
	/// A function for preparing data by a coordinator.
	int Prepare (chomp::multiwork::mwData &data);

	/// A function for accepting results by a coordinator.
	int Accept (chomp::multiwork::mwData &data);

	/// A function for taking rejected data by a coordinator.
	int Reject (chomp::multiwork::mwData &data);

	// --- Parameters of Computations ---

	/// The control number to recognize compatible workers.
	int controlNumber;

	/// A file to append the results to.
	std::ofstream f;

	/// Should the results file be flushed after every write?
	bool flushfile;

	/// The minimal value of the parameter 'a' to investigate.
	double paramMin;

	/// The maximal value of the parameter 'a' to investigate.
	double paramMax;

	/// Should the entire intervals of parameters be processed?
	bool intervals;

	/// The minimal value of delta if fixed.
	double deltaMin;

	/// The maximal value of delta if fixed.
	double deltaMax;

	/// The resolution with which lambda must be found.
	double resolution;

	/// The minimal value of lambda if requested.
	double lambdaMin;

	/// The maximal value of lambda if requested.
	double lambdaMax;

	/// The map type name.
	std::string mapName;

	/// The partition type name.
	std::string partName;

	/// The minimal size of the partition.
	int partCountMin;

	/// The maximal size of the partition.
	int partCountMax;

	// --- Current Data of Computations ---

	/// Reads the previously computed results from the input file.
	/// Returns true if successful, false if no results file exists.
	bool readResults (const char *filename);

	/// Opens a file with the results for appending or creates the file.
	/// If this is a new file then some introductory info is written.
	void openResults (const char *filename, bool newFile);

	/// The currently processed level.
	int currentLevel;

	/// The current element at the given level.
	int currentItem;

	/// The starting subdivision level.
	int startLevel;

	/// The final subdivision level.
	int finalLevel;

	/// Is there just one single parameter combination to be processed?
	bool singleItem;

	/// Should delta be computed?
	int computeDelta;

	/// Should delta0 be computed? (with paths in consideration)
	int computeDelta0;

	/// Should lambda be computed?
	int computeLambda;

	/// Should C be computed?
	int computeC;

	/// Should lambda0 be computed?
	int computeLambda0;

	/// Should the computations be rigorous?
	int rigorous;

	/// Should a sparse graph algorithm be used? [0=no, 1=yes, -1=auto]
	int sparseGraph;

	/// The bit fields for each level.
	std::vector<bitVector *> processed;

	/// The low precision for writing real numbers to files.
	int loprec;

	/// The high precision for writing real numbers to files.
	int hiprec;

	/// Computes the number of items at the given level.
	int numberOfItems (int level) const;

	/// Updates the processet bit vectors to the current level.
	void updateProcessed (int level);

}; /* class Coordinator */

// --------------------------------------------------

inline bool Coordinator::readResults (const char *filename)
{
	using namespace chomp::homology;

	// ignore a file with no or empty name
	if (!filename && !*filename)
		return false;

	// open the file for reading
	std::ifstream in (filename);
	if (!in)
		return false;

	// process all the lines with the results and mark which are read
	ignorecomments (in);
	while (in. peek () != EOF)
	{
		if (in. peek () != '*')
		{
			ignoreline (in);
			ignorecomments (in);
			continue;
		}
		in. get ();
		ignorecomments (in);
		int cLevel = -1, cColon = -1, cItem = -1;
		in >> cLevel;
		ignorecomments (in);
		cColon = in. get ();
		ignorecomments (in);
		in >> cItem;
		if ((cLevel >= 0) && (cColon == ':') && (cItem >= 0) &&
			(cItem < numberOfItems (cLevel)))
		{
			updateProcessed (cLevel);
			processed [cLevel] -> mark (cItem);
		}
		ignoreline (in);
		ignorecomments (in);
	}
	return true;
} /* Coordinator::readResults */

inline void Coordinator::openResults (const char *filename, bool newFile)
{
	using namespace chomp::homology;

	// ignore a file with no or empty name
	if (!filename && !*filename)
		return;

	// open the results file for appending
	f. open (filename, std::ios::out | std::ios::app);
	f << "; Started on " << currenttime ();
	if (!f)
		fileerror (filename);

	// indicate the meaning of the data in each line
	if (newFile)
	{
		f << "; level:num parMin parMax k delta lambda logC lambda0 "
			"compTime\n";
	}

	return;
} /* Coordinator::openResults */

inline Coordinator::Coordinator (int _controlNumber,
	const char *_filename, bool _flushfile,
	double _paramMin, double _paramMax, bool _intervals,
	double _deltaMin, double _deltaMax, double _resolution,
	double _lambdaMin, double _lambdaMax, const char *_mapName,
	const char *_partName, int _partCountMin, int _partCountMax,
	int _startLevel, int _finalLevel,
	int _computeDelta, int _computeDelta0,
	int _computeLambda, int _computeC, int _computeLambda0,
	int _rigorous, int _sparseGraph):
	controlNumber (_controlNumber), flushfile (_flushfile),
	paramMin (_paramMin), paramMax (_paramMax), intervals (_intervals),
	deltaMin (_deltaMin), deltaMax (_deltaMax), resolution (_resolution),
	lambdaMin (_lambdaMin), lambdaMax (_lambdaMax),
	mapName (_mapName), partName (_partName),
	partCountMin (_partCountMin), partCountMax (_partCountMax),
	currentLevel (_startLevel), currentItem (0),
	startLevel (_startLevel), finalLevel (_finalLevel),
	singleItem (false),
	computeDelta (_computeDelta), computeDelta0 (_computeDelta0),
	computeLambda (_computeLambda), computeC (_computeC),
	computeLambda0 (_computeLambda0), rigorous (_rigorous),
	sparseGraph (_sparseGraph),
	loprec (9), hiprec (15)
{
	using namespace chomp::homology;

	// check the correctness of levels
	if ((startLevel < 0) || (finalLevel < startLevel))
		throw "Incorrect start/final levels requested.";

	// if there is only one parameter to consider, reset the levels
	if ((paramMin == paramMax) && (deltaMin == deltaMax) &&
		(lambdaMin == lambdaMax) && (partCountMin == partCountMax))
	{
		singleItem = true;
		currentLevel = finalLevel = startLevel = 0;
	}

	// if the parameter of the equation is single, set no intervals
	if (paramMin == paramMax)
		intervals = false;

	// process and open the results file if its file name is given
	if (_filename && *_filename)
	{
		// read the previously computed results of the computations
		bool resultsRead = this -> readResults (_filename);
		if (!resultsRead)
		{
			sout << "Note: Could not open the results file. "
				"Any previous results will be ignored.\n";
		}

		// open the results file for appending
		openResults (_filename, !resultsRead);
	}

	// otherwise, warn the user
	else
		sout << "Warning: Results will not be saved to a file.";

	return;
} /* Coordinator::Coordinator */

inline Coordinator::~Coordinator ()
{
	using namespace chomp::homology;

	f << "; Finished on " << currenttime ();
	f. close ();
	for (std::vector<bitVector *>::iterator it = processed. begin ();
		it != processed. end (); ++ it)
	{
		delete *it;
	}
	return;
} /* Coordinator::~Coordinator */

inline int Coordinator::numberOfItems (int level) const
{
	if (singleItem)
		return 1;
	else if (!intervals && (level == startLevel))
		return (1 << level) + 1;
	else
		return 1 << level;
} /* Coordinator::numberOfItems */

inline void Coordinator::updateProcessed (int level)
{
	while (processed. size () <= static_cast<unsigned> (level))
	{
		int nItems = numberOfItems (processed. size ());
		processed. push_back (new bitVector (nItems));
	}
	return;
} /* Coordinator::updateProcessed */

inline int Coordinator::Prepare (chomp::multiwork::mwData &data)
{
	using namespace chomp::homology;
	using namespace chomp::multiwork;

	// reset the data to prepare
	data. Reset ();

	// compute the current values of parameters to send
	int nItems = 0;
	while (1)
	{
		// make sure the current item is set to an unmarked bit
		updateProcessed (currentLevel);
		nItems = numberOfItems (currentLevel);
		if (currentItem < nItems)
		{
			currentItem = processed [currentLevel] ->
				findUnmarked (currentItem, nItems);
		}

		// if odd items only are processed, move to the next one
		if ((currentItem < nItems) && !intervals &&
			(currentLevel != startLevel) && !(currentItem & 1))
		{
			++ currentItem;
			continue;
		}

		// if an unmarked bit was found, then prepare the data
		if (currentItem < nItems)
		{
			processed [currentLevel] -> mark (currentItem);
			break;
		}

		// if the entire final level has been processed, finish
		if (currentLevel == finalLevel)
			return mwNoData;

		// take the next level to process
		++ currentLevel;
		currentItem = 0;
	}

	// prepare the parameters of computations to send
	double paramMin1 = (paramMin == paramMax) ? paramMin : (paramMin +
		currentItem * (paramMax - paramMin) / (1 << currentLevel));
	double paramMax1 = intervals ? (paramMin + (currentItem + 1) *
		(paramMax - paramMin) / (1 << currentLevel)) : paramMin1;
	double delta1 = (deltaMin == deltaMax) ? deltaMin :
		exp (log (deltaMin) + currentItem * (log (deltaMax) -
		log (deltaMin)) / (1 << currentLevel));
	double lambda1 = (lambdaMin == lambdaMax) ? lambdaMin : (lambdaMin +
		currentItem * (lambdaMax - lambdaMin) / (1 << currentLevel));
	int partCount1 = (partCountMin == partCountMax) ? partCountMin :
		static_cast <int> ((partCountMin +
		static_cast<double> (currentItem) *
		(partCountMax - partCountMin) / (1 << currentLevel)));

	// prepare data to send
	data << currentLevel;
	data << currentItem;
	data << mapName;
	data << partName;
	data << partCount1;
	data << paramMin1;
	data << paramMax1;
	data << delta1;
	data << resolution;
	data << lambda1;
	data << computeDelta;
	data << computeDelta0;
	data << computeLambda;
	data << computeC;
	data << computeLambda0;
	data << rigorous;
	data << sparseGraph;
	data << loprec;
	data << hiprec;
	data << controlNumber;

	// show a message on what has been done
	sout << "+ " << currentLevel << ":" << currentItem << ".\n";

	++ currentItem;
	return mwOk;
} /* Coordinator::Prepare */

inline int Coordinator::Accept (chomp::multiwork::mwData &data)
{
	using namespace chomp::homology;
	using namespace chomp::multiwork;

	// decode the number
	int cLevel = 0, cItem = 0;
	data >> cLevel;
	data >> cItem;
	double pMin = 0;
	double pMax = 0;
	data >> pMin;
	data >> pMax;
	int partCount = 0;
	data >> partCount;
	double deltaBad = 0;
	double delta = 0;
	double lambda = 0;
	double logC = 0;
	double lambda0 = 0;
	data >> deltaBad;
	data >> delta;
	data >> lambda;
	data >> logC;
	data >> lambda0;
	double compTime = 0;
	data >> compTime;
	int ctrl = 0;
	data >> ctrl;
	if (ctrl != controlNumber)
	{
		f << "! Wrong data received.\n";
		throw "Wrong data received from a worker.";
	}

	// say what has been received
	sout << "* " << cLevel << ":" << cItem << " [" << pMin << "," <<
		pMax << "] k=" << partCount << " d=" << delta <<
		" l=" << lambda << " lC=" << logC << " l0=" << lambda0 <<
		" (" << compTime << "s)" << "\n";

	// save the data to the output file
	f << "* " << cLevel << ":" << cItem << " " <<
		pMin << " " << pMax << " " << partCount << " " <<
		std::setprecision (hiprec) << delta << " " <<
		lambda << " " << logC << " " << lambda0 << " " <<
		std::setprecision (loprec) << compTime << "\n";

	// flush the data file if requested to
	if (flushfile)
		f << std::flush;

	return mwOk;
} /* Coordinator::Accept */

inline int Coordinator::Reject (chomp::multiwork::mwData &data)
{
	using namespace chomp::homology;
	using namespace chomp::multiwork;

	// decode the number
	int cLevel = 0, cItem = 0;
	data >> cLevel;
	data >> cItem;

	// say that data has been rejected
	sout << "Data " << cLevel << ":" << cItem << " has been rejected.\n";
	throw "Data rejected by a worker.";

	return mwOk;
} /* Coordinator::Reject */


} // namespace unifexp

#endif // _coord_h_

/// @}

