/////////////////////////////////////////////////////////////////////////////
///
/// \file
///
/// A program for the simplicial homology computation.
///
/////////////////////////////////////////////////////////////////////////////

// Copyright (C) 2009-2011 by Pawel Pilarczyk.
//
// This file is part of my research software package. This is free software:
// you can redistribute it and/or modify it under the terms of the GNU
// General Public License as published by the Free Software Foundation,
// either version 3 of the License, or (at your option) any later version.
//
// This software is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this software; see the file "license.txt". If not,
// please, see <http://www.gnu.org/licenses/>.

// Started on March 24, 2009. Last revision: April 4, 2011.


// include some standard C++ header files
#include <istream>
#include <ostream>

// include selected header files from the CHomP library
#include "chomp/system/config.h"
#include "chomp/system/textfile.h"
#include "chomp/system/timeused.h"
#include "chomp/system/arg.h"
#include "chomp/struct/hashsets.h"

// include relevant local header files
#include "chaincon/simplex.h"
#include "comphom.h"


// --------------------------------------------------
// -------------------- OVERTURE --------------------
// --------------------------------------------------

/// The title of the program and licensing information.
const char *title = "\
Simplicial homology computation with gradient vector field.\n\
Version 0.01 (Apr 4, 2011). Copyright (C) 1997-2011 by Pawel Pilarczyk.\n\
This is free software. No warranty. Consult 'license.txt' for details.";

/// Brief help information on the program's usage.
const char *helpinfo = "\
This program computes simplicial homology groups and homology generators.\n\
The ring of coefficients is set to Z_2, the integers modulo 2.\n\
Call with:\n\
filename - the name of a file that contains a list of simplicies,\n\
Switches and additional arguments:\n\
-dpi, -dincl, -dphi - display the computed maps: pi, incl, phi,\n\
-aN - homology algorithm: 0 = old (very slow), 1 = new without additional\n\
optimization (relatively fast), 2 = new (default), 3 = using the SNF.\n\
--verify - do additional verification of the computed maps,\n\
--log filename - save the output to a file (without progress indicators),\n\
--quiet - suppress data output to the screen (whcih can be still logged),\n\
--help - display this brief help information only and exit.\n\
For more information please consult the accompanying documentation\n\
or ask the program's author at http://www.PawelPilarczyk.com/.";


// --------------------------------------------------
// ---------------------- main ----------------------
// --------------------------------------------------

/// The main procedure of the program.
/// Returns: 0 = Ok, -1 = Error, 1 = Help displayed, 2 = Wrong arguments.
int main (int argc, char *argv [])
{
	using namespace chomp::homology;

	// turn on a message that will appear if the program does not finish
	program_time = "Aborted after";

	// prepare user-configurable data
	char *filename = 0;
	bool displayPi = false;
	bool displayIncl = false;
	bool displayPhi = false;
	bool verify = false;
	int algorithmVersion = 2;

	// analyze the command line
	arguments a;
	arg (a, NULL, filename);
	arg (a, "a", algorithmVersion);
	argswitch (a, "dpi", displayPi, true);
	argswitch (a, "dincl", displayIncl, true);
	argswitch (a, "dphi", displayPhi, true);
	argswitch (a, "-verify", verify, true);
	arghelp (a);

	argstreamprepare (a);
	int argresult = a. analyze (argc, argv);
	argstreamset ();

	// show the program's title
	if (argresult >= 0)
		sout << title << '\n';

	// if something was incorrect, show an additional message and exit
	if (argresult < 0)
	{
		sout << "Call with '--help' for help.\n";
		return 2;
	}

	// if help requested or no filename present, show help information
	if ((argresult > 0) || !filename)
	{
		sout << helpinfo << '\n';
		return 1;
	}

	// try running the main function and catch an error message if thrown
	try
	{
		// prepare the data for storing the result of computation
		typedef tSimplex<int_t> CellT;
		chomp::homology::hashedset<CellT> H;
		tCombLinMap<CellT, CellT> pi, incl, phi;

		// compute the homology groups and generators
		computeHomology<CellT> (filename, H, pi, incl, phi,
			displayPi, displayIncl, displayPhi,
			verify, algorithmVersion);

		program_time = "Total time used:";
		program_time = 1;
		return 0;
	}
	catch (const char *msg)
	{
		sout << "ERROR: " << msg << '\n';
		return -1;
	}
	catch (const std::exception &e)
	{
		sout << "ERROR: " << e. what () << '\n';
		return -1;
	}
	catch (...)
	{
		sout << "ABORT: An unknown error occurred.\n";
		return -1;
	}
} /* main */

