/////////////////////////////////////////////////////////////////////////////
///
/// \file
///
/// A filtered cell complex.
///
/////////////////////////////////////////////////////////////////////////////

// Copyright (C) 2009-2011 by Pawel Pilarczyk.
//
// This file is part of my research software package. This is free software:
// you can redistribute it and/or modify it under the terms of the GNU
// General Public License as published by the Free Software Foundation,
// either version 3 of the License, or (at your option) any later version.
//
// This software is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this software; see the file "license.txt". If not,
// please, see <http://www.gnu.org/licenses/>.

// Started on March 24, 2009. Last revision: April 4, 2011.


#ifndef _CHAINCON_FILTCOMPLEX_H_
#define _CHAINCON_FILTCOMPLEX_H_


// include some standard C++ header files
#include <istream>
#include <ostream>

// include selected header files from the CHomP library
#include "chomp/system/config.h"
#include "chomp/system/textfile.h"
#include "chomp/struct/hashsets.h"


// --------------------------------------------------
// ------------- filtered cell complex --------------
// --------------------------------------------------

/// A filtered complex. Cells must be added to this complex
/// in an order that ensures that any cell that appears in the boundary
/// of another cell is added AFTER that cell.
/// Contains a method for generating the boundaries of all the cells
/// that were previously added to the complex.
/// Then the cells can be accessed in the reversed order using operator [].
template <class CellT>
class tFilteredComplex
{
public:
	/// The default constructor.
	tFilteredComplex ();

	/// Adds a cell to the complex. Does not add its boundary cells.
	/// Returns the number of the added cell in the complex.
	int_t add (const CellT &c);

	/// Returns the size of the complex.
	int_t size () const;

	/// Returns the information on whether the complex is empty or not.
	bool empty () const;

	/// Returns the given cell in the complex in the reversed order.
	const CellT &operator [] (int_t n) const;

	/// Swaps the data with another filtered complex.
	void swap (tFilteredComplex<CellT> &fc);

private:
	/// The set of all the cells in the complex in the right order.
	chomp::homology::hashedset<CellT> cells;

}; /* class tFilteredComplex */

// --------------------------------------------------

template <class CellT>
inline tFilteredComplex<CellT>::tFilteredComplex ():
	cells (1024)
{
	return;
} /* tFilteredComplex::tFilteredComplex */

template <class CellT>
inline int_t tFilteredComplex<CellT>::add (const CellT &c)
{
	return cells. add (c);
} /* tFilteredComplex::add */

template <class CellT>
inline int_t tFilteredComplex<CellT>::size () const
{
	return cells. size ();
} /* tFilteredComplex::size */

template <class CellT>
inline bool tFilteredComplex<CellT>::empty () const
{
	return cells. empty ();
} /* tFilteredComplex::empty */

template <class CellT>
inline const CellT &tFilteredComplex<CellT>::operator [] (int_t n) const
{
	int_t size = cells. size ();
	if (n >= size)
		throw "Wrong number of a cell in a filtered complex.";
	return cells [size - n - 1];
} /* tFilteredComplex::operator [] */

template <class CellT>
inline void tFilteredComplex<CellT>::swap (tFilteredComplex<CellT> &fc)
{
	cells. swap (fc. cells);
} /* tFilteredComplex::swap */


// --------------------------------------------------

template <class CellT>
inline void addBoundaries (tFilteredComplex<CellT> &K)
{
	for (int_t n = 0; n < K. size (); ++ n)
	{
		const CellT &c = K [K. size () - n - 1];
		int len = c. boundaryLength ();
		for (int i = 0; i < len; ++ i)
		{
			CellT bc (c, i);
			int_t cn = K. add (bc);
			if (cn < n)
			{
				throw "Boundary cell appears too early "
					"in the filter.";
			}
		}
	}
	return;
} /* addBoundaries */

// --------------------------------------------------

/// Writes a filtered cell complex to an output stream in the text format.
template <class CellT>
std::ostream &operator << (std::ostream &out,
	const tFilteredComplex<CellT> &C)
{
	int_t size = C. size ();
	for (int_t i = 0; i < size; ++ i)
	{
		out << C [i] << "\n";
	}
	return out;
} /* operator << */

/// Reads a filtered cell complex from an input stream.
/// The cells are supposed to be listed in the input stream
/// in the reverse order, that is, each cell should be followed
/// by its boundary cells, which is not verified in this procedure.
/// Reading the cells is interrupted in case of the empty cell
/// (which is added to the complex if the empty cell is in use)
/// or when the end of the stream is encountered.
template <class CellT>
std::istream &operator >> (std::istream &in, tFilteredComplex<CellT> &K)
{
	chomp::homology::ignorecomments (in);
	while (!in. eof ())
	{
		CellT s;
		in >> s;
		if (s. dim () < 0)
		{
#ifndef NO_EMPTY_CELL
			K. add (s);
#endif
			break;
		}
		K. add (s);
	}
	return in;
} /* operator >> */


#endif // _CHAINCON_FILTCOMPLEX_H_

