/////////////////////////////////////////////////////////////////////////////
///
/// \file
///
/// A combinatorial tensor (for coefficients in Z_2).
///
/////////////////////////////////////////////////////////////////////////////

// Copyright (C) 2009-2011 by Pawel Pilarczyk.
//
// This file is part of my research software package. This is free software:
// you can redistribute it and/or modify it under the terms of the GNU
// General Public License as published by the Free Software Foundation,
// either version 3 of the License, or (at your option) any later version.
//
// This software is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this software; see the file "license.txt". If not,
// please, see <http://www.gnu.org/licenses/>.

// Started on March 24, 2009. Last revision: April 4, 2011.


#ifndef _CHAINCON_COMBTENSOR_H_
#define _CHAINCON_COMBTENSOR_H_


// include some standard C++ header files
#include <istream>
#include <ostream>

// include selected header files from the CHomP library
#include "chomp/system/config.h"

// include relevant local header files
#include "chaincon/pair.h"
#include "chaincon/combchain.h"


// --------------------------------------------------
// -------------- combinatorial tensor --------------
// --------------------------------------------------

/// Combinatorial tensor of cells. This is in fact a list of pairs of cells
/// of the given type. One can add products of chains to the tensor,
/// which is then decomposed into tensors of individual cells and reduced.
template <class CellT>
class tCombTensor
{
public:
	/// The type of cells in the combinatorial tensor.
	typedef CellT CellType;

	/// The default constructor of an empty combinatorial tensor.
	tCombTensor ();

	/// Returns the number of elements in the combinatorial tensor.
	int_t size () const;

	/// Returns true if and only if the tensor is zero (empty set).
	bool empty () const;

	/// Returns the n-th left element of the combinatorial tensor.
	const CellT &left (int_t n) const;

	/// Returns the n-th right element of the combinatorial tensor.
	const CellT &right (int_t n) const;

	/// Adds a pair of cells to the combinatorial tensor.
	void add (const CellT &c1, const CellT &c2);

	/// Adds a pair of chains to the combinatorial tensor.
	void add (const tCombChain<CellT> &c1, const tCombChain<CellT> &c2);

	/// Adds a given tensor to the combinatorial tensor.
	void add (const tCombTensor<CellT> &ch);

//	/// Compares if the two tensors are equal.
//	bool operator == (const tCombTensor<CellT> &ch) const;

	/// Swaps the data with another tensor.
	void swap (tCombTensor<CellT> &ch);

private:
	/// The set of elements in the combinatorial tensor.
	tCombChain<tPair<CellT,CellT> > tensor;

}; /* class tCombTensor */

// --------------------------------------------------

template <class CellT>
inline tCombTensor<CellT>::tCombTensor ()
{
	return;
} /* tCombTensor::tCombTensor */

template <class CellT>
inline int_t tCombTensor<CellT>::size () const
{
	return tensor. size ();
} /* tCombTensor::size */

template <class CellT>
inline bool tCombTensor<CellT>::empty () const
{
	return tensor. empty ();
} /* tCombTensor::empty */

template <class CellT>
inline const CellT &tCombTensor<CellT>::left (int_t n) const
{
	return tensor [n]. left;
} /* tCombTensor::left */

template <class CellT>
inline const CellT &tCombTensor<CellT>::right (int_t n) const
{
	return tensor [n]. right;
} /* tCombTensor::right */

template <class CellT>
inline void tCombTensor<CellT>::add (const CellT &c1, const CellT &c2)
{
	tensor. add (tPair<CellT,CellT> (c1, c2));
} /* tCombTensor::add */

template <class CellT>
inline void tCombTensor<CellT>::add (const tCombChain<CellT> &ch1,
	const tCombChain<CellT> &ch2)
{
	int_t ch1size = ch1. size ();
	int_t ch2size = ch2. size ();
	for (int_t i = 0; i < ch1size; ++ i)
	{
		const CellT &c1 = ch1 [i];
		for (int_t j = 0; j < ch2size; ++ j)
		{
			const CellT &c2 = ch2 [j];
			tensor. add (tPair<CellT,CellT> (c1, c2));
		}
	}
	return;
} /* tCombTensor::add */

template <class CellT>
inline void tCombTensor<CellT>::add (const tCombTensor<CellT> &ch)
{
	tensor. add (ch. tensor);
	return;
} /* tCombTensor::add */


template <class CellT>
inline void tCombTensor<CellT>::swap (tCombTensor<CellT> &ch)
{
	tensor. swap (ch. cells);
	return;
} /* tCombTensor::swap */

// --------------------------------------------------

/// Writes a combinatorial tensor to an output stream.
template <class CellT>
std::ostream &operator << (std::ostream &out, const tCombTensor<CellT> &c)
{
	int_t size = c. size ();
	for (int_t i = 0; i < size; ++ i)
	{
		if (i)
			out << " + ";
		out << c. left (i) << " x " << c. right (i);
		}
	return out;
} /* operator << */

/// Reads a combinatorial tensor from an input stream.
template <class CellT>
std::istream &operator >> (std::istream &in, tCombTensor<CellT> &c)
{
	throw "Operator >> not implemented for tCombTensor.";
	return in;
} /* operator >> */

// --------------------------------------------------

/// Adds the boundary of a given tensor to the provided tensor.
template <class CellT>
inline void computeBoundary (const tCombTensor<CellT> &c,
	tCombTensor<CellT> &b)
{
	int_t size = c. size ();
	for (int_t i = 0; i < size; ++ i)
	{
		tCombChain<CellT> left;
		computeBoundary (c. left (i), left);
		tCombChain<CellT> right;
		computeBoundary (c. right (i), right);
		b. add (left, right);
	}
	return;
} /* computeBoundary */

/// Returns the boundary of a given tensor.
template <class CellT>
inline tCombTensor<CellT> boundary (const tCombTensor<CellT> &c)
{
	// prepare a tensor that is going to contain the boundary of c
	tCombTensor<CellT> b;

	// add the boundary of each pair in the tensor c to the tensor b
	computeBoundary (c, b);

	// return the computed boundary tensor
	return b;
} /* boundary */


#endif // _CHAINCON_COMBTENSOR_H_

