/////////////////////////////////////////////////////////////////////////////
///
/// \file
///
/// A combinatorial chain, that is, a chain with Z_2 coefficients.
///
/////////////////////////////////////////////////////////////////////////////

// Copyright (C) 2009-2011 by Pawel Pilarczyk.
//
// This file is part of my research software package. This is free software:
// you can redistribute it and/or modify it under the terms of the GNU
// General Public License as published by the Free Software Foundation,
// either version 3 of the License, or (at your option) any later version.
//
// This software is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this software; see the file "license.txt". If not,
// please, see <http://www.gnu.org/licenses/>.

// Started on March 24, 2009. Last revision: March 6, 2011.


#ifndef _CHAINCON_COMBCHAIN_H_
#define _CHAINCON_COMBCHAIN_H_


// include some standard C++ header files
#include <istream>
#include <ostream>

// include selected header files from the CHomP library
#include "chomp/system/config.h"
#include "chomp/struct/hashsets.h"


// --------------------------------------------------
// -------------- combinatorial chain ---------------
// --------------------------------------------------

/// A combinatorial chain. This is in fact a finite set of cells.
/// It corresponds to chains with coefficients in F_2 (a.k.a. Z_2).
template <class CellT>
class tCombChain
{
public:
	/// The type of cells in the combinatorial chain.
	typedef CellT CellType;

	/// The default constructor of an empty combinatorial chain.
	tCombChain ();

	/// The constructor of a combinatorial chain containing one cell.
	explicit tCombChain (const CellT &c);

	/// Returns the number of elements in the combinatorial chain.
	int_t size () const;

	/// Returns true if and only if the chain is zero (empty set).
	bool empty () const;

	/// Returns the n-th element of the combinatorial chain.
	const CellT &operator [] (int_t n) const;

	/// Returns the position of the given cell in the chain
	/// or -1 if the cell is not there.
	int_t position (const CellT &c) const;

	/// Checks if the given cell appears in the chain.
	bool contains (const CellT &c) const;

	/// Adds a given cell to the combinatorial chain.
	void add (const CellT &c);

	/// Adds a given chain to the combinatorial chain.
	void add (const tCombChain<CellT> &ch);

	/// Compares if the two chains are equal.
	bool operator == (const tCombChain<CellT> &ch) const;

	/// Swaps the data with another chain.
	void swap (tCombChain<CellT> &ch);

private:
	/// The set of elements in the combinatorial chain.
	chomp::homology::hashedset<CellT> cells;

}; /* class tCombChain */

// --------------------------------------------------

template <class CellT>
inline tCombChain<CellT>::tCombChain ():
	cells (1)
{
	return;
} /* tCombChain::tCombChain */

template <class CellT>
inline tCombChain<CellT>::tCombChain (const CellT &c):
	cells (1)
{
	cells. add (c);
	return;
} /* tCombChain::tCombChain */

template <class CellT>
inline int_t tCombChain<CellT>::size () const
{
	return cells. size ();
} /* tCombChain::size */

template <class CellT>
inline bool tCombChain<CellT>::empty () const
{
	return cells. empty ();
} /* tCombChain::empty */

template <class CellT>
inline const CellT &tCombChain<CellT>::operator [] (int_t n) const
{
	return cells [n];
} /* tCombChain::operator [] */

template <class CellT>
inline int_t tCombChain<CellT>::position (const CellT &c) const
{
	return cells. getnumber (c);
} /* tCombChain::position */

template <class CellT>
inline bool tCombChain<CellT>::contains (const CellT &c) const
{
	return (cells. getnumber (c) >= 0);
} /* tCombChain::contains */

template <class CellT>
inline void tCombChain<CellT>::add (const CellT &c)
{
	int_t n = cells. getnumber (c);
	if (n < 0)
		cells. add (c);
	else
		cells. removenum (n);
	return;
} /* tCombChain::add */

template <class CellT>
inline void tCombChain<CellT>::add (const tCombChain<CellT> &ch)
{
	int_t size = ch. size ();
	for (int_t i = 0; i < size; ++ i)
		this -> add (ch [i]);
	return;
} /* tCombChain::add */

template <class CellT>
inline bool tCombChain<CellT>::operator == (const tCombChain<CellT> &ch)
	const
{
	int_t size = cells. size ();
	int_t chSize = ch. cells. size ();
	if (size != chSize)
		return false;
	for (int_t i = 0; i < size; ++ i)
	{
		if (!ch. cells. check (cells [i]))
			return false;
	}
	return true;
} /* tCombChain::operator == */

template <class CellT>
inline void tCombChain<CellT>::swap (tCombChain<CellT> &ch)
{
	cells. swap (ch. cells);
	return;
} /* tCombChain::swap */

// --------------------------------------------------

/// Writes a combinatorial chain to an output stream.
template <class CellT>
std::ostream &operator << (std::ostream &out, const tCombChain<CellT> &c)
{
	int_t size = c. size ();
	for (int_t i = 0; i < size; ++ i)
	{
		if (i)
			out << "+";
		out << c [i];
	}
	return out;
} /* operator << */

/// Reads a combinatorial chain from an input stream.
template <class CellT>
std::istream &operator >> (std::istream &in, tCombChain<CellT> &c)
{
	throw "Operator >> not implemented for tCombChain.";
	return in;
} /* operator >> */

// --------------------------------------------------

template <class CellT>
tCombChain<CellT> operator + (const tCombChain<CellT> &a,
	const tCombChain<CellT> &b)
{
	tCombChain<CellT> c (a);
	c. add (b);
	return c;
} /* operator + */

// --------------------------------------------------

/// Adds the boundary of a given cell to the provided chain.
template <class CellT>
inline void computeBoundary (const CellT &c, tCombChain<CellT> &b)
{
	// determine the length of the boundary of this cell
	int length = c. boundaryLength ();

	// process all the cells in its boundary
	for (int i = 0; i < length; ++ i)
	{
		// create the subsequent boundary cell
		CellT bc (c, i);

		// add the boundary cell to the given chain
		b. add (bc);
	}
	return;
} /* computeBoundary */

/// Returns the boundary of a given cell.
template <class CellT>
inline tCombChain<CellT> boundary (const CellT &c)
{
	// prepare a chain that is going to contain the boundary of c
	tCombChain<CellT> b;

	// add the boundary of the cell to b
	computeBoundary (c, b);

	// return the computed boundary chain
	return b;
} /* boundary */

/// Adds the boundary of a given chain to the provided chain.
template <class CellT>
inline void computeBoundary (const tCombChain<CellT> &c,
	tCombChain<CellT> &b)
{
	int_t size = c. size ();
	for (int_t i = 0; i < size; ++ i)
		computeBoundary (c [i], b);
	return;
} /* computeBoundary */

/// Returns the boundary of a given chain.
template <class CellT>
inline tCombChain<CellT> boundary (const tCombChain<CellT> &c)
{
	// prepare a chain that is going to contain the boundary of c
	tCombChain<CellT> b;

	// add the boundary of each cell in the chain to b
	computeBoundary (c, b);

	// return the computed boundary chain
	return b;
} /* boundary */


#endif // _CHAINCON_COMBCHAIN_H_

