/////////////////////////////////////////////////////////////////////////////
///
/// \file
///
/// A cubical version of the Alexander-Whitney diagonal.
///
/////////////////////////////////////////////////////////////////////////////

// Copyright (C) 2009-2011 by Pawel Pilarczyk.
//
// This file is part of my research software package. This is free software:
// you can redistribute it and/or modify it under the terms of the GNU
// General Public License as published by the Free Software Foundation,
// either version 3 of the License, or (at your option) any later version.
//
// This software is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this software; see the file "license.txt". If not,
// please, see <http://www.gnu.org/licenses/>.

// Started on March 24, 2009. Last revision: March 6, 2011.


#ifndef _CHAINCON_AWDIAGCUB_H_
#define _CHAINCON_AWDIAGCUB_H_


// include some standard C++ header files
#include <istream>
#include <ostream>

// include selected header files from the CHomP library
#include "chomp/system/config.h"

// include relevant local header files
#include "chaincon/cubcell.h"
#include "chaincon/combtensor.h"


// --------------------------------------------------
// ------------------ AW diagonal -------------------
// --------------------------------------------------

/// Computes the Alexander-Whitney diagonal of a cubical cell
/// using the formula from Serre's paper.
/// Please, note that this formula has only been implemented
/// for 2-dimensional cells at the moment; the general formula
/// will be implemented soon.
template <class CoordT>
inline tCombTensor<tCubCell<CoordT> > AWdiagonal (const tCubCell<CoordT> &c)
{
	// abort the program if the dimension is different than two
	if (c. dim () != 2)
		throw "A-W diagonal implemented for 2-dim cubes only.";

	// extract the coordinates of the four vertices of the cubical cell
	int dim = c. spaceDim ();
	CoordT coord [4] [chomp::homology::MaxBasDim];
	int indices [2];
	int index = 0;
	for (int i = 0; i < dim; ++ i)
	{
		coord [0] [i] = c. left (i);
		coord [3] [i] = c. right (i);
		if (coord [0] [i] != coord [3] [i])
		{
			coord [1] [i] = index ? coord [3] [i] :
				coord [0] [i];
			coord [2] [i] = index ? coord [0] [i] :
				coord [3] [i];
			indices [index ++] = i;
		}
		else
		{
			coord [1] [i] = coord [0] [i];
			coord [2] [i] = coord [0] [i];
		}
	}

	// create the products of respective cells
	tCombTensor<tCubCell<CoordT> > tensor;
	tensor. add (tCubCell<CoordT> (dim, coord [0], coord [2]),
		tCubCell<CoordT> (dim, coord [2], coord [3]));
	tensor. add (tCubCell<CoordT> (dim, coord [0], coord [1]),
		tCubCell<CoordT> (dim, coord [1], coord [3]));

	return tensor;
} /* AWdiagonal */


#endif // _CHAINCON_AWDIAGCUB_H_

