/////////////////////////////////////////////////////////////////////////////
///
/// \file
///
/// A generic procedure for the Alexander-Whithey diagonal computation
/// for 2-dimensional homology generators.
/// This is a complete procedure good for simplicial, cubical,
/// or other complexes.
///
/////////////////////////////////////////////////////////////////////////////

// Copyright (C) 2009-2011 by Pawel Pilarczyk.
//
// This file is part of my research software package. This is free software:
// you can redistribute it and/or modify it under the terms of the GNU
// General Public License as published by the Free Software Foundation,
// either version 3 of the License, or (at your option) any later version.
//
// This software is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this software; see the file "license.txt". If not,
// please, see <http://www.gnu.org/licenses/>.

// Started on March 24, 2009. Last revision: April 3, 2011.


#ifndef _AWDIAG2D_H_
#define _AWDIAG2D_H_


// include selected header files from the CHomP library
#include "chomp/system/config.h"
#include "chomp/system/textfile.h"
#include "chomp/struct/hashsets.h"

// include relevant local header files
#include "chaincon/awdiag.h"
#include "chaincon/combchain.h"
#include "chaincon/combtensor.h"
#include "chaincon/comblinmap.h"


// --------------------------------------------------
// ---------------- compute homology ----------------
// --------------------------------------------------

/// Computes the Alexander-Whithey diagonal for 2-dimensional
/// homology generators.
/// Uses the projection and inclusion maps computed previously
/// as parts of a chain contraction of a cell complex.
template <class CellT>
void computeAWdiagonal (const chomp::homology::hashedset<CellT> &H,
	const tCombLinMap<CellT, CellT> &pi,
	const tCombLinMap<CellT, CellT> &incl,
	bool displayDiag, bool verify)
{
	using chomp::homology::sout;

	// compute the AW decomposition of each 2D homology generator
	int_t Hsize = H. size ();
	for (int_t i = 0; i < Hsize; ++ i)
	{
		if (H [i]. dim () != 2)
			continue;
		tCombChain<CellT> homGen = incl (H [i]);
		tCombTensor<CellT> diag (AWdiagonal (homGen));
		if (displayDiag)
		{
			sout << "A-W diagonal of the cycle "
				"corresponding to " << H [i] << ":\n";
			int_t size = homGen. size ();
			for (int_t j = 0; j < size; ++ j)
			{
				tCombTensor<CellT> diagGen (AWdiagonal
					(tCombChain<CellT> (homGen [j])));
				sout << homGen [j] << " -> " <<
					diagGen << "\n";
			}
		//	sout << "which reduces to:\n" << diag << "\n";
		}
		int_t n = diag. size ();
		tCombTensor<CellT> diag1dim;
		for (int_t j = 0; j < n; ++ j)
		{
			if (diag. left (j). dim () != 1)
				continue;
			if (diag. right (j). dim () != 1)
				continue;
			diag1dim. add (diag. left (j), diag. right (j));
		}
		tCombTensor<CellT> diagBoundary;
		if (verify)
			computeBoundary (diag1dim, diagBoundary);
		if (displayDiag)
		{
			sout << "restricted to 1-dim:\n" << diag1dim << "\n";
			if (verify)
			{
				sout << "its boundary: " <<
					diagBoundary << "\n";
			}
		}
		if (verify)
		{
			if (diagBoundary. empty ())
				sout << "Verified: bd AW " << H [i] <<
					" = 0.\n";
			else
				sout << "Failed to verify that bd AW " <<
					H [i] << " = 0.\n";
		}
		tCombTensor<CellT> diagHom (pi (diag1dim));
		sout << "A-W decomp of " << H [i] << ": " << diagHom << "\n";
	}

	return;
} /* computeAWdiagonal */


#endif // _AWDIAG2D_H_

