#include <iostream>
#include <string>
#include <vector>
#include <algorithm>

#include "chomp/system/config.h"
#include "chomp/system/textfile.h"
using chomp::homology::sout;
using chomp::homology::scon;
using chomp::homology::slog;
using chomp::homology::sbug;
using chomp::homology::fileerror;
using chomp::homology::commandline;
using chomp::homology::currenttime;
using chomp::homology::ignorecomments;
#include "chomp/system/timeused.h"
using chomp::homology::timeused;
using chomp::homology::program_time;
#include "chomp/system/arg.h"
using chomp::homology::arguments;
using chomp::homology::arg;
using chomp::homology::setstreams;
#include "chomp/struct/hashsets.h"
using chomp::homology::hashedset;
#include "chomp/struct/multitab.h"
using chomp::homology::multitable;
#include "chomp/cubes/pointbas.h"
using chomp::homology::tPointBase;
using chomp::homology::NumBits;
using chomp::homology::NumMask;
using chomp::homology::MaxBasDim;

// for the 3rd homology algorithm, using SNF (Feb 8, 2011):
// include a hacked version of the official chains.h
//#include "chomp/homology/chains.h"
#include "awchains.h"
using chomp::homology::chaincomplex;
using chomp::homology::chain;
using chomp::homology::mmatrix;
#include "chomp/struct/integer.h"
using chomp::homology::integer;


// --------------------------------------------------
// -------------------- OVERTURE --------------------
// --------------------------------------------------

/// The title of the program and licensing information.
const char *title = "\
Alexander-Whitney diagonal computation for cavities in a 3D cubical set.\n\
Version 0.01 (Feb 14, 2011). Copyright (C) 1997-2011 by Pawel Pilarczyk.\n\
This is free software. No warranty. Consult 'license.txt' for details.";

/// Brief help information on the program's usage.
const char *helpinfo = "\
This program computes the Alexander-Whitney diagonal for cavities\n\
in a 3D full cubical set. It is an illustration of the method\n\
introduced in my paper written with A. Berciano, H. Molina, A. Pacheco,\n\
and P. Real.\n\
Call with:\n\
filename - the name of a file that contains the definition of the input,\n\
Switches and additional arguments:\n\
-s - assume the input is a simplicial complex,\n\
-c - assume the input is a cubical complex (this is by default),\n\
-h - compute homology only, don't compute the A-W diagonal,\n\
-dpi, -dincl, -dphi - display the computed maps: pi, incl, phi,\n\
-ddiag - display the A-W diagonal of each 2D homology generator,\n\
-v - do additional verification of the computed maps,\n\
-tN - run test computation no. N (temporary, for debugging etc.),\n\
-aN - A-W diagonal version no. N: 0 = simpl subdiv (default),\n\
\t1 = paper, 2-3 other formulas, 4 - based on Serre's paper,\n\
-mN - homology algorithm: 0 = old, 1 = new, 2 = newer (default),\n\
\t3,4 = using Smith diagonalization and an AM model,\n\
--help - display this brief help information only and exit.\n\
For more information please consult the accompanying documentation\n\
or ask the program's author at http://www.PawelPilarczyk.com/.";


// --------------------------------------------------
// ---------------- compute homology ----------------
// --------------------------------------------------

/// Computes the homology of a cellular complex.
/// The cellular complex is read from the given file.
/// Only top-dimensional cells should be listed in the file,
/// or otherwise the filtration may not be generated correctly.
/// If requested, the homology is only computed, without AW diagonal.
template <class CellT>
void compAWdiag (const char *filename, bool homologyOnly, bool verify,
	bool displayPi, bool displayIncl, bool displayPhi,
	bool displayDiag)
{
	// read the filtered cell complex
	tFilteredComplex<CellT> K;
	{
		sout << "Reading '" << filename << "'... ";
		std::ifstream in (filename);
		if (!in)
			fileerror (filename);
		while (!in. eof ())
		{
			CellT s;
			in >> s;
			if (s. dim () < 0)
				break;
			K. add (s);
		}
		in. close ();
		sout << K. size () << " cells read.\n";
	}
//	sout << "Cells read from the input:\n" << K << "\n";
	sout << "Adding boundaries... ";
	K. addBoundaries ();
	sout << K. size () << " cells total.\n";
//	sout << "Filtered complex:\n" << K << "==========\n";

	// compute the homology together with the chain contraction
	hashedset<CellT> H;
	tCombLinMap<CellT, CellT> pi, incl, phi;
	computeHomology (K, H, pi, incl, phi);

	// show the computed maps
	if (displayPi)
		sout << "The projection map pi:\n" << pi;
	if (displayIncl)
		sout << "The inclusion map incl:\n" << incl;
	if (displayPhi)
		sout << "The homology gradient vector field phi:\n" << phi;

	// show the computed homology representants, ordered by dimension
	sout << "Homology representants (ordered by dimension):\n";
	int_t Hsize = H. size ();
	int_t displayed = 0;
	for (int dim = 0; displayed < Hsize; ++ dim)
	{
		for (int_t i = 0; i < Hsize; ++ i)
		{
			if (H [i]. dim () != dim)
				continue;
			sout << "gen_" << (displayed ++) << ": " <<
				H [i] << "\n";
		}
	}

	// prepare the boundary map, to be used for verifications
	tCombLinMap<CellT, CellT> boundaryMap;

	// do several verifications to make sure the computed maps are good
	if (verify)
	{
		generateBoundaryMap (K, boundaryMap);
		if (phi * boundaryMap * phi == phi)
			sout << "Verified: 'phi bd phi = phi'.\n";
		else
			sout << "Failed to verify that "
				"'phi pd phi = phi'.\n";

		if (boundaryMap * phi * boundaryMap == boundaryMap)
			sout << "Verified: 'bd phi bd = bd'.\n";
		else
			sout << "Failed to verify that 'bd phi pd = bd'.\n";

		tCombLinMap<CellT, CellT> zeroMap;
		if (phi * phi == zeroMap)
			sout << "Verified: 'phi phi = 0'.\n";
		else
			sout << "Failed to verify that 'phi phi = 0'.\n";

		tCombLinMap<CellT, CellT> composition (boundaryMap * phi);
		composition. add (phi * boundaryMap);
		addIdentity (K, composition);
		if (incl * pi == composition)
			sout << "Verified: 'i pi = id - bd phi - phi bd'.\n";
		else
			sout << "Failed to verify that "
				"'i pi = id - bd phi - phi bd'.\n";

		tCombLinMap<CellT, CellT> idH;
		addIdentity (H, idH);
		if (pi * incl == idH)
			sout << "Verified: 'pi i = id_H'.\n";
		else
			sout << "Failed to verify that 'pi i = id_H'.\n";

		if (pi * phi == zeroMap)
			sout << "Verified: 'pi phi = 0'.\n";
		else
			sout << "Failed to verify that 'pi phi = 0'.\n";

		if (phi * incl == zeroMap)
			sout << "Verified: 'phi i = 0'.\n";
		else
			sout << "Failed to verify that 'phi i = 0'.\n";
	}

	// quit if the homology only had to be computed
	if (homologyOnly)
		return;

	// compute the AW decomposition of each 2D homology generator
	for (int_t i = 0; i < Hsize; ++ i)
	{
		if (H [i]. dim () != 2)
			continue;
		tCombChain<CellT> homGen = incl (H [i]);
		tCombTensor<CellT> diag (AWdiagonal (homGen));
		if (displayDiag)
		{
			sout << "A-W diagonal of the cycle "
				"corresponding to " << H [i] << ":\n";
			int_t size = homGen. size ();
			for (int_t j = 0; j < size; ++ j)
			{
				tCombTensor<CellT> diagGen (AWdiagonal
					(tCombChain<CellT> (homGen [j])));
				sout << homGen [j] << " -> " <<
					diagGen << "\n";
			}
		//	sout << "which reduces to:\n" << diag << "\n";
		}
		int_t n = diag. size ();
		tCombTensor<CellT> diag1dim;
		for (int_t j = 0; j < n; ++ j)
		{
			if (diag. left (j). dim () != 1)
				continue;
			if (diag. right (j). dim () != 1)
				continue;
			diag1dim. add (diag. left (j), diag. right (j));
		}
		if (displayDiag)
		{
			sout << "restricted to 1-dim:\n" << diag1dim << "\n";
			if (verify)
			{
				sout << "its boundary: " <<
					boundaryMap (diag1dim) << "\n";
			}
		}
		if (verify)
		{
			if (boundaryMap (diag1dim). empty ())
				sout << "Verified: bd AW " << H [i] <<
					" = 0.\n";
			else
				sout << "Failed to verify that bd AW " <<
					H [i] << " = 0.\n";
		}
		tCombTensor<CellT> diagHom (pi (diag1dim));
		sout << "A-W decomp of " << H [i] << ": " << diagHom << "\n";
	}

	return;
} /* compAWdiag */


// --------------------------------------------------
// --------------------- tests ----------------------
// --------------------------------------------------

/// Runs an elementary test of some features
/// of the class template "tSimplex".
void testSimplex ()
{
	sout << "TEST: Simplex.\n";
	std::string a ("w1"), b ("w2"), c ("w3"), d ("w4");
	std::vector<std::string> vec1;
	vec1. push_back (a);
	vec1. push_back (b);
	vec1. push_back (c);
//	vec1. push_back (d);
	tSimplex<std::string> s1 (vec1. size () - 1, vec1);

	sout << "s1 = " << s1 << ".\n";

	sout << "Boundary " << s1 << " = ";
	for (int i = 0; i < s1. boundaryLength (); ++ i)
	{
		tSimplex<std::string> s2 (s1, i);
		sout << ((s1. boundaryCoef (i) > 0) ? "+" : "-");
		sout << s2;
	}
	sout << " = " << boundary (s1);
	sout << ".\n";

	sout << "Delta_AW " << s1 << " = " << AWdiagonal (s1) << ".\n";

	return;
} /* testSimplex */

/// Runs an elementary test of some features
/// of the class template "tCubCell".
void testCubCell ()
{
	sout << "TEST: CubCell.\n";
	int coordLeft [] = {10, 0, 20};
	int coordRight [] = {11, 0, 21};
//	sout << "Enter a cubical cell as a product of intervals:\n";
	tCubCell<int> c (3, coordLeft, coordRight);
//	std::cin >> c;

	sout << "c = " << c << ".\n";

	sout << "Boundary " << c << " = ";
	for (int i = 0; i < c. boundaryLength (); ++ i)
	{
		tCubCell<int> b (c, i);
		sout << ((c. boundaryCoef (i) > 0) ? "+" : "-");
		sout << b;
	}
	sout << ".\n";

	sout << "Delta_AW " << c << " = " << AWdiagonal (c) << ".\n";

	return;
} /* testCubCell */

/// Runs a basic test of some features
/// of the linear combinatorial map class.
void testMap ()
{
	sout << "TEST: Map.\n";
	std::string a ("a"), b ("b"), c ("c"), d ("d");
	std::vector<std::string> vec1;
	vec1. push_back (a);
	vec1. push_back (b);
	vec1. push_back (c);
//	vec1. push_back (d);
	tSimplex<std::string> s1 (vec1. size () - 1, vec1);

	tCombLinMap<tSimplex<std::string>,tSimplex<std::string> > f;
	std::vector<tSimplex<std::string> > cells;
	cells. push_back (s1);
	generateBoundaryMap (cells, f);
	sout << f;
	return;
} /* testMap */

/// Runs a basic test of selected features of a filtered cell complex.
void testComplex ()
{
	sout << "TEST: Complex.\n";
	std::string a ("a"), b ("b"), c ("c"), d ("d");
	std::vector<std::string> vec1;
	vec1. push_back (a);
	vec1. push_back (b);
	vec1. push_back (c);
//	vec1. push_back (d);
	tSimplex<std::string> s1 (vec1. size () - 1, vec1);
	tFilteredComplex<tSimplex<std::string> > K;
	K. add (s1);
	K. addBoundaries ();
	sout << "Filtered complex:\n" << K << "==========\n";

	return;
} /* testComplex */

/// Runs a very basic test of homology computation.
void testHomology ()
{
	sout << "TEST: Homology.\n";
	std::string a ("a"), b ("b"), c ("c"), d ("d");
	std::vector<std::string> vec1;
	vec1. push_back (a);
	vec1. push_back (b);
	vec1. push_back (c);
//	vec1. push_back (d);
	tSimplex<std::string> s1 (vec1. size () - 1, vec1);
	tFilteredComplex<tSimplex<std::string> > K;
	for (int i = 0; i < s1. boundaryLength (); ++ i)
		K. add (tSimplex<std::string> (s1, i));
	K. addBoundaries ();
	hashedset<tSimplex<std::string> > H;
	tCombLinMap<tSimplex<std::string>,tSimplex<std::string> >
		pi, incl, phi;
	sout << "Filtered complex:\n" << K << "==========\n";

	computeHomology (K, H, pi, incl, phi);
	sout << "Homology representants:\n";
	int_t Hsize = H. size ();
	for (int_t i = 0; i < Hsize; ++ i)
		sout << "gen_" << i << " = " << H [i] << "\n";
	sout << "The projection map pi:\n" << pi;
	sout << "The inclusion map incl:\n" << incl;
	sout << "The homology gradient vector field phi:\n" << phi;

	return;
} /* testHomology */


// --------------------------------------------------
// ---------------------- main ----------------------
// --------------------------------------------------

int main (int argc, char *argv [])
// Return: 0 = Ok, -1 = Error, 1 = Help displayed, 2 = Wrong arguments.
{
	// turn on a message that will appear if the program does not finish
	program_time = "Aborted after";

	// prepare user-configurable data
	char *filename = 0;
	bool simplicial = false;
	bool homologyOnly = false;
	bool verify = false;
	bool displayPi = false;
	bool displayIncl = false;
	bool displayPhi = false;
	bool displayDiag = false;
	int testComput = 0;

	// analyze the command line
	arguments a;
	arg (a, NULL, filename);
	arg (a, "t", testComput);
	arg (a, "m", homAlgVersion);
	argswitch (a, "s", simplicial, true);
	argswitch (a, "c", simplicial, false);
	argswitch (a, "h", homologyOnly, true);
	argswitch (a, "dpi", displayPi, true);
	argswitch (a, "dincl", displayIncl, true);
	argswitch (a, "dphi", displayPhi, true);
	argswitch (a, "ddiag", displayDiag, true);
	argswitch (a, "v", verify, true);
	arghelp (a);

	argstreamprepare (a);
	int argresult = a. analyze (argc, argv);
	argstreamset ();

	// show the program's title
	if (argresult >= 0)
		sout << title << '\n';

	// if something was incorrect, show an additional message and exit
	if (argresult < 0)
	{
		sout << "Call with '--help' for help.\n";
		return 2;
	}

	// if help requested or no filename present, show help information
	if ((argresult > 0) || (!testComput && !filename))
	{
		sout << helpinfo << '\n';
		return 1;
	}

	// try running the main function and catch an error message if thrown
	try
	{
		typedef void (*testType) (void);
		const int testCount = 5;
		testType testFunction [testCount] = {testSimplex,
			testCubCell, testMap, testComplex, testHomology};
		if (testComput)
		{
			if ((testComput > 0) && (testComput <= testCount))
				(testFunction [testComput - 1]) ();
			else
				sout << "WARNING: Test function number "
					"out of range. Ignored.\n";
		}
		else if (simplicial)
		{
			compAWdiag<tSimplex<int_t> > (filename, homologyOnly,
				verify, displayPi, displayIncl, displayPhi,
				displayDiag);
		}
		else
		{
			compAWdiag<tCubCell<int_t> > (filename, homologyOnly,
				verify, displayPi, displayIncl, displayPhi,
				displayDiag);
		}
		program_time = "Total time used:";
		program_time = 1;
		return 0;
	}
	catch (const char *msg)
	{
		sout << "ERROR: " << msg << '\n';
		return -1;
	}
	catch (const std::exception &e)
	{
		sout << "ERROR: " << e. what () << '\n';
		return -1;
	}
	catch (...)
	{
		sout << "ABORT: An unknown error occurred.\n";
		return -1;
	}
} /* main */

